package org.yiwan.seiya.mybatis.dbsetup.plugin;

/*
 * The MIT License
 *
 * Copyright (c) 2006, The Codehaus
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of
 * this software and associated documentation files (the "Software"), to deal in
 * the Software without restriction, including without limitation the rights to
 * use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is furnished to do
 * so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.springframework.beans.factory.config.AutowireCapableBeanFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;
import org.springframework.context.support.FileSystemXmlApplicationContext;
import org.yiwan.seiya.mybatis.extension.entity.BaseEntity;

import java.util.Map;
import java.util.Set;

/**
 * Execute Mybatis database operation with external dataset files.
 *
 * @author <a href="mailto:tcwmj@hotmail.com">Kenny Wong</a>
 */
@Mojo(name = "import")
public class MybatisDBImportMojo extends AbstractMybatisDBSetupMojo {

    /**
     * <p>Type of Database operation to perform. default value is "REPLACE"</p>
     *
     * <p>Supported types are:</p>
     * <ul>
     * <li>INSERT</li>
     * <li>UPDATE</li>
     * <li>DELETE</li>
     * <li>REPLACE</li>
     * </ul>
     */
    @Parameter(defaultValue = "REPLACE")
    private String type;

    /**
     * <p>Array of spring dataset context URI<p/>
     */
    @Parameter(required = true)
    private String[] datasets;

    /**
     * <p>Indicates the dataset contexts located in classpath or file system</p>
     * <p>Default value is false (file system）</p>
     */
    @Parameter(defaultValue = "false")
    private boolean classpathedDatasets;

    private ApplicationContext datasetContext;

    @Override
    public void execute() throws MojoExecutionException, MojoFailureException {
        super.execute();
        loadDatasetContexts();

        Class<? extends BaseEntity> clazz;
        try {
            clazz = getBaseEntityType();
        } catch (ClassNotFoundException e) {
            getLog().error(String.format("class %s was not found", baseEntityType), e);
            return;
        }
        Map<String, ? extends BaseEntity> dataSetBeans = datasetContext.getBeansOfType(clazz);

        Set<? extends Map.Entry<String, ? extends BaseEntity>> entrySet = dataSetBeans.entrySet();
        AutowireCapableBeanFactory autowireCapableBeanFactory = applicationContext.getAutowireCapableBeanFactory();
        getLog().info("process starting");
        switch (type.toUpperCase()) {
            case "INSERT":
                processInserting(entrySet, autowireCapableBeanFactory);
                break;
            case "UPDATE":
                processUpdating(entrySet, autowireCapableBeanFactory);
                break;
            case "DELETE":
                processDeleting(entrySet, autowireCapableBeanFactory);
                break;
            case "REPLACE":
                processReplacing(entrySet, autowireCapableBeanFactory);
                break;
            default:
                getLog().info("process nothing");
        }
        getLog().info("process done");
    }

    private void processInserting(Set<? extends Map.Entry<String, ? extends BaseEntity>> entrySet,
                                  AutowireCapableBeanFactory autowireCapableBeanFactory) {
        for (Map.Entry<String, ? extends BaseEntity> entry : entrySet) {
            getLog().debug(String.format("inserting %s", entry.getKey()));
            BaseEntity<?> entryValue = entry.getValue();
            autowireCapableBeanFactory.autowireBean(entryValue);
            entryValue.insertSelective();
        }
    }

    private void processUpdating(Set<? extends Map.Entry<String, ? extends BaseEntity>> entrySet,
                                 AutowireCapableBeanFactory autowireCapableBeanFactory) {
        for (Map.Entry<String, ? extends BaseEntity> entry : entrySet) {
            getLog().debug(String.format("updating %s", entry.getKey()));
            BaseEntity<?> entryValue = entry.getValue();
            autowireCapableBeanFactory.autowireBean(entryValue);
            entryValue.updateByPrimaryKeySelective();
        }
    }

    private void processDeleting(Set<? extends Map.Entry<String, ? extends BaseEntity>> entrySet,
                                 AutowireCapableBeanFactory autowireCapableBeanFactory) {
        for (Map.Entry<String, ? extends BaseEntity> entry : entrySet) {
            getLog().debug(String.format("deleting %s", entry.getKey()));
            BaseEntity<?> entryValue = entry.getValue();
            autowireCapableBeanFactory.autowireBean(entryValue);
            entryValue.delete();
        }
    }

    private void processReplacing(Set<? extends Map.Entry<String, ? extends BaseEntity>> entrySet,
                                  AutowireCapableBeanFactory autowireCapableBeanFactory) {
        for (Map.Entry<String, ? extends BaseEntity> entry : entrySet) {
            getLog().debug(String.format("replacing %s", entry.getKey()));
            BaseEntity<?> entryValue = entry.getValue();
            autowireCapableBeanFactory.autowireBean(entryValue);
            entryValue.replace();
        }
    }

    /**
     * Load dataset contexts
     */
    private void loadDatasetContexts() {
        datasetContext = classpathedDatasets ?
            new ClassPathXmlApplicationContext(datasets) : new FileSystemXmlApplicationContext(datasets);
    }
}
