package org.yiwan.seiya.tower.test;

import com.google.common.collect.ImmutableMap;
import io.qameta.allure.Allure;
import io.restassured.RestAssured;
import io.restassured.builder.RequestSpecBuilder;
import io.restassured.builder.ResponseSpecBuilder;
import io.restassured.config.LogConfig;
import io.restassured.filter.log.LogDetail;
import io.restassured.http.ContentType;
import io.restassured.parsing.Parser;
import io.restassured.specification.RequestSpecification;
import io.restassured.specification.ResponseSpecification;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.Charsets;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.output.WriterOutputStream;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpStatus;
import org.slf4j.MDC;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.testng.AbstractTestNGSpringContextTests;
import org.testng.ITestContext;
import org.testng.ITestResult;
import org.testng.annotations.AfterClass;
import org.testng.annotations.AfterMethod;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.BeforeMethod;
import org.yiwan.seiya.tower.test.config.TestConfig;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintStream;
import java.nio.charset.Charset;
import java.util.Arrays;
import java.util.Locale;
import java.util.Map;

import static io.restassured.RestAssured.given;
import static org.hamcrest.Matchers.equalTo;
import static org.testng.ITestResult.SUCCESS;

@Slf4j
@ContextConfiguration(classes = {TestConfig.class})
public abstract class AbstractTowerTestNGSpringTests extends AbstractTestNGSpringContextTests {

    private final static String SIFTING_KEY = "ClassAndMethodName";

    private final static String USER_CENTER_LOGIN_URL = "/client/login";

    private final static String TOKEN_HEADER_NAME = "x-app-token";

    private final static String CLIENT_ID = "clientId";

    private final static String SECRET = "secret";

    protected final static String CODE = "code";

    protected RequestSpecification requestSpec;

    protected RequestSpecBuilder requestSpecBuilder;

    protected ResponseSpecification responseSpec;

    protected ResponseSpecBuilder responseSpecBuilder;

    private File logFile;

    @Autowired
    private ApplicationContext applicationContext;

    @Autowired
    private TestConfig testConfig;

    private Boolean enableGateway;

    private Boolean enableDatabaseOperation;

    public abstract String getBaseURI();

    public abstract Integer getPort();

    public abstract String getBasePath();

    /**
     * 获取bean
     *
     * @param beanName
     * @param <T>
     * @return
     */
    public <T> T getBean(String beanName) {
        return (T) applicationContext.getBean(beanName);
    }

    public <T> T createBean(Class<T> clazz) {
        return applicationContext.getAutowireCapableBeanFactory().createBean(clazz);
    }

    public <T> Map<String, T> getBeansOfType(Class<T> type) {
        return applicationContext.getBeansOfType(type);
    }

    /**
     * 国际化使用
     *
     * @param key
     * @return
     */
    public String getMessage(String key) {
        return applicationContext.getMessage(key, null, Locale.getDefault());
    }

    /**
     * 获取当前环境
     *
     * @return
     */
    public String[] getActiveProfiles() {
        return applicationContext.getEnvironment().getActiveProfiles();
    }

    /**
     * 获取当前环境
     *
     * @return
     */
    public String getActiveProfile() {
        String[] activeProfiles = getActiveProfiles();
        if (activeProfiles.length != 1) {
            throw new RuntimeException("invalid active profiles " + Arrays.toString(activeProfiles));
        }

        return activeProfiles[0];
    }

    /**
     * 是否启用网关登陆
     *
     * @return
     */
    public boolean isEnableGateway() {
        if (null == enableGateway) {
            String activeProfile = getActiveProfile();
            enableGateway = StringUtils.equalsAnyIgnoreCase(activeProfile, testConfig.getGatewayEnabledProfiles());
        }
        return enableGateway;
    }

    /**
     * 是否启用数据库操作
     *
     * @return
     */
    public boolean isEnableDatabaseOperation() {
        if (enableDatabaseOperation) {
            String activeProfile = getActiveProfile();
            enableDatabaseOperation = StringUtils.equalsAnyIgnoreCase(activeProfile, testConfig.getDatabaseOperationEnabledProfiles());
        }
        return enableDatabaseOperation;
    }

    @BeforeClass
    public void beforeClass() {
        MDC.remove(SIFTING_KEY);
        RestAssured.defaultParser = Parser.JSON;
    }

    @AfterClass
    public void afterClass() {
        RestAssured.reset();
    }

    @BeforeMethod
    public void beforeMethod(ITestContext testContext, ITestResult testResult) throws IOException {
        refreshSiftingAppender(testResult);
        logRestAssuredOutput(testResult);
        buildRequestSpec();
        buildResponseSpec();
        if (isEnableGateway()) {
            String token = obtainGatewayToken();
            requestSpecBuilder.addHeader(TOKEN_HEADER_NAME, token);
            requestSpec = requestSpecBuilder.build();
        }
    }

    @AfterMethod
    public void afterMethod(ITestContext testContext, ITestResult testResult) throws IOException {
        if (testResult.getStatus() != SUCCESS && testResult.getThrowable() != null) {
            log.error(testResult.getThrowable().getMessage(), testResult.getThrowable());
            // attach log content into allure reports
            addAllureAttachment(logFile);
        }
    }

    /**
     * sifting log
     *
     * @param testResult {@link ITestResult}
     */
    private void refreshSiftingAppender(ITestResult testResult) {
        MDC.put(SIFTING_KEY, this.getClass().getSimpleName() + File.separator + testResult.getMethod().getMethodName());
    }

    /**
     * add rest assured console log into log files
     *
     * @param testResult {@link ITestResult}
     * @throws IOException
     */
    private void logRestAssuredOutput(ITestResult testResult) throws IOException {
        File logFolder = new File("target/logs/", this.getClass().getSimpleName());
        logFolder.mkdirs();
        logFile = new File(logFolder, testResult.getMethod().getMethodName() + ".log");
        FileWriter fileWriter = new FileWriter(logFile, true);
        PrintStream printStream = new PrintStream(new WriterOutputStream(fileWriter, Charsets.UTF_8), true);
        RestAssured.config = RestAssured.config().logConfig(LogConfig.logConfig().defaultStream(printStream));
    }

    protected void buildRequestSpec() {
        requestSpecBuilder = new RequestSpecBuilder();
        requestSpecBuilder.setAccept(ContentType.JSON.withCharset(Charset.forName("UTF-8")));
        requestSpecBuilder.setContentType(ContentType.JSON.withCharset(Charset.forName("UTF-8")));
        requestSpecBuilder.setBaseUri(getBaseURI());
        requestSpecBuilder.setPort(getPort());
        requestSpecBuilder.setBasePath(getBasePath());
        requestSpecBuilder.log(LogDetail.ALL);
        requestSpec = requestSpecBuilder.build();
    }

    protected void buildResponseSpec() {
        responseSpecBuilder = new ResponseSpecBuilder();
        responseSpecBuilder.log(LogDetail.ALL);
        responseSpec = responseSpecBuilder.build();
    }

    /**
     * attach log content into allure reports
     *
     * @throws IOException
     */
    private void addAllureAttachment(File file) throws IOException {
        Allure.addAttachment("test log", FileUtils.readFileToString(file, Charsets.UTF_8));
    }

    /**
     * 登陆网关获取token
     *
     * @return
     */
    private String obtainGatewayToken() {
        Map<String, String> map = ImmutableMap.of(CLIENT_ID, testConfig.getGatewayClientId(), SECRET, testConfig.getGatewaySecret());
        return given()
            .spec(requestSpec)
            .body(map)
            .when()
            .post(USER_CENTER_LOGIN_URL)
            .then()
            .spec(responseSpec)
            .statusCode(HttpStatus.SC_OK)
            .body(CODE, equalTo(1))
            .extract()
            .path("data");
    }
}