package org.yiwan.seiya.swagger.testcodegen.plugin;

/*
 * Copyright 2001-2005 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing permissions and limitations under
 * the License.
 */

import io.swagger.models.HttpMethod;
import io.swagger.models.Operation;
import io.swagger.models.Path;
import io.swagger.models.Swagger;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.RandomStringUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.VelocityEngine;

import java.io.*;
import java.util.List;
import java.util.Map;
import java.util.Properties;

/**
 * Goal which generates client/server code from a swagger json/yaml definition.
 */
@Mojo(name = "generate", defaultPhase = LifecyclePhase.GENERATE_TEST_SOURCES)
public class SeiyaSwaggerTestcodegenMojo extends AbstractSeiyaSwaggerTestcodegenMojo {

    @Override
    public void execute() throws MojoExecutionException {
        if (skip) {
            getLog().info("swagger test code generation is skipped.");
            return;
        }
        Swagger swagger = getSwagger(inputSpec);
        Map<String, Path> swaggerPathMap = swagger.getPaths();
        initTemplate();
        generateSwaggerTestCode(swaggerPathMap);
        project.addCompileSourceRoot(sourceDir.getAbsolutePath());
    }

    private void initTemplate() {
        final Properties props = new Properties();
        props.put("resource.loader", "class");
        props.put("class.resource.loader.class", "org.apache.velocity.runtime.resource.loader.ClasspathResourceLoader");
        final VelocityEngine engine = new VelocityEngine(props);
        engine.init();
        velocityTemplate = engine.getTemplate(TEST_CODEGEN_TEMPLATE, encoding);
    }

    private void generateSwaggerTestCode(Map<String, Path> swaggerPathMap) throws MojoExecutionException {
        String superClassName = null;
        if (superClass != null) {
            String[] superClassSplits = superClass.split("\\.");
            superClassName = getLastString(superClassSplits);
        }

        String output = sourceDir.getAbsolutePath() + File.separator + StringUtils.replace(classPackage, ".", File.separator);
        File outputDir = new File(output);
        createDirectoryIfRequired(outputDir);
        createDirectoryIfRequired(resourceDir);
        String testData = null;
        try (InputStream inputStream = this.getClass().getClassLoader().getResourceAsStream(TEST_DATA_TEMPLATE)) {
            testData = IOUtils.toString(inputStream, encoding);
        } catch (IOException e) {
            throw new MojoExecutionException("Error on reading file " + TEST_DATA_TEMPLATE, e);
        }

        for (Map.Entry<String, Path> entry : swaggerPathMap.entrySet()) {
            getLog().debug("entry path = " + entry.getKey());

            Map<HttpMethod, Operation> operationMap = entry.getValue().getOperationMap();
            for (Map.Entry<HttpMethod, Operation> operationEntry : operationMap.entrySet()) {
                HttpMethod httpMethod = operationEntry.getKey();
                String action = httpMethod.name().toLowerCase();

                Operation operation = operationEntry.getValue();
                String operationId = operation.getOperationId();
                if (StringUtils.isEmpty(operationId)) {
                    operationId = RandomStringUtils.randomAlphabetic(10).toLowerCase();
                }
                String apiName = underscore(sanitizeName(operationId)).toUpperCase();
                String testClassName = String.format("%sTest", camelize(operationId));
                String outputFileName = String.format("%s.java", testClassName);
                String resourceFileName = String.format("%s.xml", testClassName);

                List<String> tags = operation.getTags();
                String apiClassName;
                if (tags.size() > 0) {
                    for (String tag : operation.getTags()) {
                        String camelizedTag = camelize(tag);
                        String uncapitalizedTag = camelize(tag, true);
                        apiClassName = String.format("%sApi", camelizedTag);
                        File taggedOutput = new File(outputDir, uncapitalizedTag);
                        createDirectoryIfRequired(taggedOutput);
                        File taggedResourceDir = new File(resourceDir, uncapitalizedTag);
                        createDirectoryIfRequired(taggedResourceDir);
                        final File outputFile = new File(taggedOutput, outputFileName);
                        if (override || !outputFile.exists()) {
//                            try (FileWriter fileWriter = new FileWriter(outputFile)) {
//                                writeContent(fileWriter, String.format("%s.%s", classPackage, uncapitalizedTag), superClassName, testClassName, apiClassName, apiName, action, uncapitalizedTag);
                            try (BufferedWriter bufferedWriter = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(outputFile), encoding))) {
                                writeContent(bufferedWriter, String.format("%s.%s", classPackage, uncapitalizedTag), superClassName, testClassName, apiClassName, apiName, action, uncapitalizedTag);
                            } catch (final IOException e) {
                                throw new MojoExecutionException("Error on creating file " + outputFile, e);
                            }
                        }
                        final File resourceFile = new File(taggedResourceDir, resourceFileName);
                        if (override || !resourceFile.exists()) {
                            try (BufferedWriter bufferedWriter = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(resourceFile), encoding))) {
                                bufferedWriter.write(testData);
                            } catch (final IOException e) {
                                throw new MojoExecutionException("Error on creating file " + resourceFile, e);
                            }
                        }
                    }
                } else {
                    apiClassName = "ApiApi";
                    final File outputFile = new File(output, outputFileName);
                    if (override || !outputFile.exists()) {
                        try (BufferedWriter bufferedWriter = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(outputFile), encoding))) {
                            writeContent(bufferedWriter, classPackage, superClassName, testClassName, apiClassName, apiName, action, null);
                        } catch (final IOException e) {
                            throw new MojoExecutionException("Error on creating file " + outputFile, e);
                        }
                    }
                    final File resourceFile = new File(resourceDir, resourceFileName);
                    if (override || !resourceFile.exists()) {
                        try (BufferedWriter bufferedWriter = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(resourceFile), encoding))) {
                            bufferedWriter.write(testData);
                        } catch (final IOException e) {
                            throw new MojoExecutionException("Error on creating file " + resourceFile, e);
                        }
                    }
                }
            }
        }
    }

    private void writeContent(final Writer writer, String classPackage, String superClassName, String testClassName, String apiClassName, String apiName, String action, String tag) {
        final VelocityContext context = new VelocityContext();
        context.put("classPackage", classPackage);
        context.put("superClass", superClass);
        context.put("superClassName", superClassName);
        context.put("testClassName", testClassName);
        context.put("apiPackage", apiPackage);
        context.put("apiClassName", apiClassName);
        context.put("apiName", apiName);
        context.put("appid", appid);
        context.put("action", action);
        context.put("ormType", ormType);
        context.put("tag", tag);
        context.put("framework", framework);
        context.put("reportType", reportType);
        velocityTemplate.merge(context, writer);
    }

}
