package org.yiwan.seiya.swagger.testcodegen.plugin;

import io.swagger.models.Swagger;
import io.swagger.parser.SwaggerCompatConverter;
import io.swagger.parser.SwaggerParser;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProject;
import org.apache.velocity.Template;

import java.io.File;
import java.io.IOException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public abstract class AbstractSeiyaSwaggerTestcodegenMojo extends AbstractMojo {

    static boolean allowUnicodeIdentifiers = false;

    final static String SWAGGER_VERSION_V2 = "2.0";

    final static String TEST_CODEGEN_TEMPLATE = "test-codegen-template.vm";

    final static String TEST_DATA_TEMPLATE = "test-data-template.xml";

    /**
     * The project being built.
     */
    @Parameter(readonly = true, required = true, defaultValue = "${project}")
    MavenProject project;

    /**
     * Skip the execution.
     */
    @Parameter(defaultValue = "false")
    Boolean skip;

    /**
     * Location of the test source directory.
     */
    @Parameter(defaultValue = "${project.build.testSourceDirectory}")
    File sourceDir;

    /**
     * Location of the test resource directory.
     */
    @Parameter(defaultValue = "${project.basedir}/src/test/resources")
    File resourceDir;

    /**
     * Location of the swagger spec, as URL or file.
     */
    @Parameter(required = true)
    String inputSpec;

    /**
     * Sets an optional override
     */
    @Parameter(defaultValue = "false")
    Boolean override;

    /**
     * class package
     */
    @Parameter(defaultValue = SWAGGER_VERSION_V2)
    String swaggerVersion;

    /**
     * class package
     */
    @Parameter(required = true)
    String classPackage;

    /**
     * api package
     */
    @Parameter(required = true)
    String apiPackage;

    /**
     * super class
     */
    @Parameter
    String superClass;

    /**
     * app id
     */
    @Parameter(required = true)
    String appid;

    /**
     * mybatis or dbunit
     */
    @Parameter(defaultValue = "mybatis")
    String ormType;

    /**
     * testng or junit
     */
    @Parameter(defaultValue = "testng")
    String framework;

    @Parameter(defaultValue = "UTF-8")
    String encoding;

    /**
     * allure or empty
     */
    @Parameter(defaultValue = "allure")
    String reportType;

    Template velocityTemplate;

    static void createDirectoryIfRequired(File file) {
        if (!file.exists()) {
            file.mkdirs();
        }
    }

    static String getLastString(String[] strings) {
        return getString(strings, -1);
    }

    static String getString(String[] strings, int offset) {
        return strings[strings.length + offset];
    }

    Swagger getSwagger(String inputSpec) {
        Swagger swagger = null;
        if (SWAGGER_VERSION_V2.equals(swaggerVersion)) {
            SwaggerParser swaggerParser = new SwaggerParser();
            swagger = swaggerParser.read(inputSpec, null, true);
        } else {
            SwaggerCompatConverter swaggerCompatConverter = new SwaggerCompatConverter();
            try {
                swagger = swaggerCompatConverter.read(inputSpec, null);
            } catch (IOException e) {
                getLog().error("cannot read api-doc from spec[version_v1.x]", e);
            }
        }
        return swagger;
    }

    /**
     * Camelize name (parameter, property, method, etc) with upper case for first letter
     * copied from Twitter elephant bird
     * https://github.com/twitter/elephant-bird/blob/master/core/src/main/java/com/twitter/elephantbird/util/Strings.java
     *
     * @param word string to be camelize
     * @return camelized string
     */
    String camelize(String word) {
        return camelize(word, false);
    }

    /**
     * Camelize name (parameter, property, method, etc)
     *
     * @param word                 string to be camelize
     * @param lowercaseFirstLetter lower case for first letter if set to true
     * @return camelized string
     */
    String camelize(String word, boolean lowercaseFirstLetter) {
        // Replace all slashes or minus with dots (package separator)
        Pattern p = Pattern.compile("[/\\- ](.?)");
        Matcher m = p.matcher(word);
        while (m.find()) {
            word = m.replaceFirst("." + m.group(1)/*.toUpperCase()*/); // FIXME: a parameter should not be assigned. Also declare the methods parameters as 'final'.
            m = p.matcher(word);
        }

        // case out dots
        String[] parts = word.split("\\.");
        StringBuilder f = new StringBuilder();
        for (String z : parts) {
            if (z.length() > 0) {
                f.append(Character.toUpperCase(z.charAt(0))).append(z.substring(1));
            }
        }
        word = f.toString();

        m = p.matcher(word);
        while (m.find()) {
            word = m.replaceFirst("" + Character.toUpperCase(m.group(1).charAt(0)) + m.group(1).substring(1)/*.toUpperCase()*/);
            m = p.matcher(word);
        }

        // Uppercase the class name.
        p = Pattern.compile("(\\.?)(\\w)([^\\.]*)$");
        m = p.matcher(word);
        if (m.find()) {
            String rep = m.group(1) + m.group(2).toUpperCase() + m.group(3);
            rep = rep.replaceAll("\\$", "\\\\\\$");
            word = m.replaceAll(rep);
        }

        // Remove all underscores
        p = Pattern.compile("(_)(.)");
        m = p.matcher(word);
        while (m.find()) {
            word = m.replaceFirst(m.group(2).toUpperCase());
            m = p.matcher(word);
        }

        if (lowercaseFirstLetter && word.length() > 0) {
            word = word.substring(0, 1).toLowerCase() + word.substring(1);
        }

        return word;
    }

    /**
     * Underscore the given word.
     * Copied from Twitter elephant bird
     * https://github.com/twitter/elephant-bird/blob/master/core/src/main/java/com/twitter/elephantbird/util/Strings.java
     *
     * @param word The word
     * @return The underscored version of the word
     */
    String underscore(String word) {
        String firstPattern = "([A-Z]+)([A-Z][a-z])";
        String secondPattern = "([a-z\\d])([A-Z])";
        String replacementPattern = "$1_$2";
        // Replace package separator with slash.
        word = word.replaceAll("\\.", "/"); // FIXME: a parameter should not be assigned. Also declare the methods parameters as 'final'.
        // Replace $ with two underscores for inner classes.
        word = word.replaceAll("\\$", "__");
        // Replace capital letter with _ plus lowercase letter.
        word = word.replaceAll(firstPattern, replacementPattern);
        word = word.replaceAll(secondPattern, replacementPattern);
        word = word.replace('-', '_');
        word = word.toLowerCase();
        return word;
    }

    String sanitizeName(String name) {
        // NOTE: performance wise, we should have written with 2 replaceAll to replace desired
        // character with _ or empty character. Below aims to spell out different cases we've
        // encountered so far and hopefully make it easier for others to add more special
        // cases in the future.

        // better error handling when map/array type is invalid
        if (name == null) {
            getLog().error("String to be sanitized is null. Default to ERROR_UNKNOWN");
            return "ERROR_UNKNOWN";
        }

        // if the name is just '$', map it to 'value' for the time being.
        if ("$".equals(name)) {
            return "value";
        }

        // input[] => input
        name = name.replaceAll("\\[\\]", ""); // FIXME: a parameter should not be assigned. Also declare the methods parameters as 'final'.

        // input[a][b] => input_a_b
        name = name.replaceAll("\\[", "_");
        name = name.replaceAll("\\]", "");

        // input(a)(b) => input_a_b
        name = name.replaceAll("\\(", "_");
        name = name.replaceAll("\\)", "");

        // input.name => input_name
        name = name.replaceAll("\\.", "_");

        // input-name => input_name
        name = name.replaceAll("-", "_");

        // input name and age => input_name_and_age
        name = name.replaceAll(" ", "_");

        // remove everything else other than word, number and _
        // $php_variable => php_variable
        if (allowUnicodeIdentifiers) { //could be converted to a single line with ?: operator
            name = Pattern.compile("\\W", Pattern.UNICODE_CHARACTER_CLASS).matcher(name).replaceAll("");
        } else {
            name = name.replaceAll("\\W", "");
        }

        return name;
    }
}
