package com.xforceplus.phoenix.split.service.dataflow.impl;

import com.google.common.collect.Lists;
import com.xforceplus.phoenix.split.constant.ErrorAmountPolicyEnum;
import com.xforceplus.phoenix.split.constant.InvoiceItemOrder;
import com.xforceplus.phoenix.split.constant.TaxDeviceType;
import com.xforceplus.phoenix.split.constant.TaxInvoiceSourceEnum;
import com.xforceplus.phoenix.split.domain.ItemGroup;
import com.xforceplus.phoenix.split.domain.RuleInfo;
import com.xforceplus.phoenix.split.domain.SplitGroupLimit;
import com.xforceplus.phoenix.split.exception.SplitBizException;
import com.xforceplus.phoenix.split.exception.SplitRuleParamException;
import com.xforceplus.phoenix.split.model.BillInfo;
import com.xforceplus.phoenix.split.model.BillItem;
import com.xforceplus.phoenix.split.model.ItemTypeCodeEnum;
import com.xforceplus.phoenix.split.model.SplitRule;
import com.xforceplus.phoenix.split.service.SplitRuleUtil;
import com.xforceplus.phoenix.split.service.dataflow.DataProcessPlugin;
import com.xforceplus.phoenix.split.service.dataflow.SpecialInvoiceService;
import com.xforceplus.phoenix.split.service.dataflow.impl.invoiceLimit.ItemOrderCombineItemService;
import com.xforceplus.phoenix.split.service.dataflow.impl.invoiceLimit.MinInvoiceCombineItemService;
import com.xforceplus.phoenix.split.util.BillItemUtils;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;
import java.util.stream.Collectors;

/**
 * 预制发票分组
 */
@Service
public class InvoiceLimitProcessPlugin implements DataProcessPlugin {

    private static final Logger logger = LoggerFactory.getLogger(InvoiceLimitProcessPlugin.class);
    public static final String ERROR_AMOUNT = "0.01";

    /**
     * 预制发票限制
     */
    @Value("${split.invoice.count.limit.each:1000}")
    private int splitInvoiceCount = 1000;

    /**
     * 单张发票误差累计最大值
     */
    @Value("#{${invoiceMaxErrorAmountPolicy}}")
    private Map<String, BigDecimal> invoiceMaxErrorAmountPolicy;
    @Autowired
    private MinPackagePlugin2 minPackagePlugin;
    /**
     * 单张发票误差累计最大值
     */
    @Value("#{${calculationConfig}}")
    private Map<String, String> calculationConfig;
    @Autowired
    private SpecialInvoiceService specialInvoiceService;
    @Autowired
    private ItemOrderCombineItemService itemOrderCombineItemService;
    @Autowired
    private MinInvoiceCombineItemService minInvoiceCombineItemService;

    @Override
    public List<ItemGroup> processData(List<ItemGroup> itemGroups, BillInfo billInfo, RuleInfo rule) {

        return processData(itemGroups, billInfo, rule, TaxDeviceType.HX_SINGL);
    }

    @Override
    public List<ItemGroup> processData(List<ItemGroup> itemGroups, BillInfo billInfo, RuleInfo rule, TaxDeviceType taxDeviceType) {

        /**
         * 1.校验拆票规则
         * 2.遍历拆分后的发票
         * 3.处理数电特殊发票
         * 4.处理普通发票
         * 4.1根据明细顺序处理
         *
         */
        SplitRule splitRule = rule.getSplitRule();
        SplitRuleUtil.validateSplitRule(splitRule);

        List<ItemGroup> allItemGroups = Lists.newArrayListWithExpectedSize(itemGroups.size());
        for (ItemGroup itemGroup : itemGroups) {

            assembleBillItems(allItemGroups, itemGroup, billInfo, splitRule, taxDeviceType);
            if (allItemGroups.size() > splitInvoiceCount) {
                throw new SplitBizException("预估单次拆分预制发票数量超过限制数量:" + splitInvoiceCount);
            }
        }

        return allItemGroups;
    }

    /**
     * @param allItemGroups 总明细组
     * @param itemGroup     待合并明细组
     * @param billInfo      业务单主信息
     * @param splitRule     拆票规则
     * @param taxDeviceType 设备类型
     */
    private void assembleBillItems(List<ItemGroup> allItemGroups, ItemGroup itemGroup, BillInfo billInfo, SplitRule splitRule, TaxDeviceType taxDeviceType) {

        List<BillItem> wholeBillItems = itemGroup.getBillItems();
        String invoiceOfGroupKey = UUID.randomUUID().toString();
        String itemTypeCode = wholeBillItems.get(0).getItemTypeCode();

        /**
         * 全电电票&&特殊票种 与 普通票种分开成票
         */
        String taxInvoiceSource = splitRule.getTaxInvoiceSource();
        if (TaxInvoiceSourceEnum.QD.getValue().equals(taxInvoiceSource)) {

            List<BillItem> specialBillItems = wholeBillItems.stream().filter(e -> ItemTypeCodeEnum.isAllElectronicSpecialInvoice(e.getItemTypeCode())).collect(Collectors.toList());

            if (CollectionUtils.isNotEmpty(specialBillItems)) {
                logger.info("InvoiceLimitProcessPlugin.processData.assembleSpecialItems.billItems:{}", specialBillItems);

                allItemGroups.addAll(specialInvoiceService.assembleItems(splitRule, itemGroup, billInfo, specialBillItems));
                wholeBillItems.removeAll(specialBillItems);
                itemGroup.setBillItems(wholeBillItems);
            }
        }

        if (CollectionUtils.isEmpty(wholeBillItems)) {
            return;
        }

        SplitGroupLimit splitGroupLimit = SplitRuleUtil.createSplitGroupLimit(splitRule, itemTypeCode, billInfo, wholeBillItems.get(0));
        BigDecimal defaultErrorAmount = getMaxErrorAmountByTaxDeviceType(taxDeviceType, splitRule);

        if (InvoiceItemOrder.ITEM_NO_ORDER == splitGroupLimit.getInvoiceItemOrder() ||
                InvoiceItemOrder.ITEM_NO_MINIMUM_INVOICES == splitGroupLimit.getInvoiceItemOrder()) {

            splitGroupLimit.setInvoiceItemOrder(InvoiceItemOrder.ITEM_NO_ORDER);
            allItemGroups.addAll(this.orderSplitItemGroup(itemGroup, splitGroupLimit, invoiceOfGroupKey, taxDeviceType, splitRule));

        } else if (InvoiceItemOrder.MINIMUM_INVOICES == splitGroupLimit.getInvoiceItemOrder()) {

            splitGroupLimit.setInvoiceItemOrder(InvoiceItemOrder.MINIMUM_INVOICES);
            /**
             * 不存在负数请情况
             */
            boolean isNegative = itemGroup.getBillItems().stream().anyMatch(item -> item.getAmountWithTax().compareTo(BigDecimal.ZERO) <= 0);
            if (!isNegative && Objects.nonNull(splitGroupLimit.getInvoiceMaxErrorAmount()) &&
                    defaultErrorAmount.compareTo(splitGroupLimit.getInvoiceMaxErrorAmount()) != 0) {
                allItemGroups.addAll(this.miniPlugin(itemGroup, splitGroupLimit, invoiceOfGroupKey));
            } else {
                allItemGroups.addAll(this.leastSplitItemGroup(itemGroup, splitGroupLimit, invoiceOfGroupKey, taxDeviceType, splitRule));
            }

        } else {
            throw new SplitRuleParamException("发票明细顺序规则有误");
        }
    }

    protected BigDecimal getMaxErrorAmountByTaxDeviceType(TaxDeviceType taxDeviceType, SplitRule splitRule) {

        boolean isQdSpecialAddition = splitRule.isEnableAdvancedValidation();
        if (isQdSpecialAddition) {
            logger.info("isQdSpecialAddition policyBigDecimal:1.27");
            return BigDecimal.valueOf(1.27);
        }

        BigDecimal policyBigDecimal =  (invoiceMaxErrorAmountPolicy != null && invoiceMaxErrorAmountPolicy.containsKey(taxDeviceType.getCode())) ? invoiceMaxErrorAmountPolicy.get(taxDeviceType.getCode()) : BigDecimal.valueOf(1.26);
        logger.info("policyBigDecimal:{}", policyBigDecimal);
        return policyBigDecimal;
    }

    private List<ItemGroup> orderSplitItemGroup(ItemGroup itemGroup, SplitGroupLimit splitGroupLimit, String invoiceOfGroupKey, TaxDeviceType taxDeviceType, SplitRule splitRule) {
        List<ItemGroup> processedItemGroups = Lists.newLinkedList();

        logger.debug("itemGroup.getBillItems size:{}", itemGroup.getBillItems().size());
        while (CollectionUtils.isNotEmpty(itemGroup.getBillItems())) {
            List<BillItem> processedItemList = itemOrderCombineItemService.combineItem(itemGroup.getBillItems(), splitGroupLimit, taxDeviceType, splitRule);
            if (CollectionUtils.isEmpty(processedItemList)) {
                throw new SplitBizException("请检查拆票规则的配置是否正确");
            }
            ItemGroup processedItemGroup = new ItemGroup(itemGroup.getParentGroupFlag());
            processedItemGroup.setInvoiceOfGroupKey(invoiceOfGroupKey);
            processedItemGroup.setBillItems(processedItemList);
            processedItemGroups.add(processedItemGroup);
        }
        return processedItemGroups;
    }

    private List<ItemGroup> miniPlugin(ItemGroup itemGroup, SplitGroupLimit splitGroupLimit, String invoiceOfGroupKey) {

        List<ItemGroup> processedItemGroups = Lists.newLinkedList();
        List<List<BillItem>> result = minPackagePlugin.processData(itemGroup.getBillItems(), splitGroupLimit);
        if (CollectionUtils.isNotEmpty(result)) {
            for (List<BillItem> processedItemList : result) {
                ItemGroup processedItemGroup = new ItemGroup(itemGroup.getParentGroupFlag());
                processedItemGroup.setInvoiceOfGroupKey(invoiceOfGroupKey);
                processedItemGroup.setBillItems(processedItemList);
                processedItemGroups.add(processedItemGroup);
            }
            return processedItemGroups;
        }
        return Collections.EMPTY_LIST;
    }

    private List<ItemGroup> leastSplitItemGroup(ItemGroup itemGroup, SplitGroupLimit splitGroupLimit, String invoiceOfGroupKey, TaxDeviceType taxDeviceType, SplitRule splitRule) {
        List<ItemGroup> allItemGroups = Lists.newLinkedList();

        List<BillItem> billItems = itemGroup.getBillItems();
        Map<Boolean, List<BillItem>> isSplitItemMap = billItems.stream().collect(Collectors.groupingBy(BillItem::isSplitItem));

        List<BillItem> processItems = new LinkedList<>();
        //未拆分过金额的明细,需要处理分组
        List<BillItem> isNotSplitBillItems = isSplitItemMap.get(false);
        if (CollectionUtils.isNotEmpty(isNotSplitBillItems)) {
            processItems.addAll(isNotSplitBillItems);
        }
        //拆分过金额的明细,每张单据最后一条需要处理分组
        List<BillItem> isSplitBillItems = isSplitItemMap.get(true);

        List<BillItem> noNeedProcessItems = Lists.newLinkedList();

        //对拆分的明细的处理，按业务单明细id分组，取金额最小的一条明细，加入processItems
        //是因为拆分的明细里，剩余的金额最小的，一定是拆剩下的；业务单明细id下其他剩余的1条或多条明细，一定是符合顶额拆分规则的（个别可能因为数量取整反算）
        if (CollectionUtils.isNotEmpty(isSplitBillItems)) {
            Map<String, List<BillItem>> billIdItemsMap = isSplitBillItems.stream().collect(Collectors.groupingBy(BillItem::getSalesbillItemId));
            for (Map.Entry<String, List<BillItem>> stringListEntry : billIdItemsMap.entrySet()) {
                List<BillItem> items = stringListEntry.getValue();
                /**
                 * 原意为找到最后一条明细，重算进行合票操作（最后一条明细是金额最小的那一条）
                 * 修改的原因为: 其他plugin中可能会把顺序修改，最后一条不一定为金额最小的那个
                 */
                // BillItem item = items.get(items.size() - 1);
                BillItem item = BillItemUtils.findMinAmountBillItem(items);

                items.remove(item);
                processItems.add(item);
                noNeedProcessItems.addAll(items);
            }
            for (BillItem noNeedProcessItem : noNeedProcessItems) {
                ItemGroup notNeedProcessItemGroup = new ItemGroup(itemGroup.getParentGroupFlag());
                notNeedProcessItemGroup.setInvoiceOfGroupKey(invoiceOfGroupKey);
                notNeedProcessItemGroup.setBillItems(Lists.newArrayList(noNeedProcessItem));
                allItemGroups.add(notNeedProcessItemGroup);
            }
        }

        if (CollectionUtils.isNotEmpty(processItems)) {
            //按照金额倒排
            processItems = processItems.stream()
                    .sorted(((Comparator<BillItem>) (o1, o2) -> {
                        BigDecimal value1 = actualAmount(o1, splitGroupLimit.isLimitIsAmountWithTax());
                        BigDecimal value2 = actualAmount(o2, splitGroupLimit.isLimitIsAmountWithTax());
                        return value1.compareTo(value2);
                    }).reversed())
                    .collect(Collectors.toList());
            logger.debug("itemGroup.processItems size:{}", processItems.size());

            List<List<BillItem>> invoices = this.minInvoice(processItems, splitGroupLimit, taxDeviceType, splitRule);
            for (List<BillItem> invoiceItems : invoices) {
                ItemGroup newItemGroup = new ItemGroup(itemGroup.getParentGroupFlag());
                newItemGroup.setInvoiceOfGroupKey(invoiceOfGroupKey);
                newItemGroup.setBillItems(invoiceItems);
                allItemGroups.add(newItemGroup);
            }
        }

        return allItemGroups;
    }

    protected List<List<BillItem>> minInvoice(List<BillItem> processItems, SplitGroupLimit splitGroupLimit, TaxDeviceType taxDeviceType, SplitRule splitRule) {
        List<List<BillItem>> result = new ArrayList<>();
        while (CollectionUtils.isNotEmpty(processItems)) {
            List<BillItem> processedItemList = minInvoiceCombineItemService.combineItem(processItems, splitGroupLimit, taxDeviceType, splitRule);
            if (CollectionUtils.isEmpty(processedItemList)) {
                throw new SplitBizException("请检查拆票规则的配置是否正确");
            }
            result.add(processedItemList);
        }
        return result;
    }

    /**
     * 明细实际金额
     * @param item                 单据明细
     * @param limitIsAmountWithTax
     * @return 明细实际金额
     */
    protected BigDecimal actualAmount(BillItem item, boolean limitIsAmountWithTax) {
        BigDecimal result;

        if (limitIsAmountWithTax) {
            result = item.getAmountWithTax().subtract(item.getDiscountWithTax());
        } else {
            result = item.getAmountWithoutTax().subtract(item.getDiscountWithoutTax());
        }

        return result;
    }

    /**
     * 计算票面税额差是否超过业务配置数值
     * @param totalErrorAmount
     * @param currentItemErrorAmount
     * @param businessInvoiceMaxErrorAmount 业务配置的税额差，可能为空
     * @return
     */
    public Boolean totalErrorAmountCalculationForBusiness(BigDecimal totalErrorAmount, BigDecimal currentItemErrorAmount, BigDecimal businessInvoiceMaxErrorAmount) {

        return businessInvoiceMaxErrorAmount != null &&
                businessInvoiceMaxErrorAmount.compareTo(BigDecimal.ZERO) != 0 &&
                totalErrorAmount.add(currentItemErrorAmount).abs().compareTo(businessInvoiceMaxErrorAmount) > 0;
    }

    /**
     * 单条明细计算尾差 考虑到百旺航信的误差调整
     * @param item
     * @return
     */
    protected BigDecimal errorAmountCalculation(BillItem item, TaxDeviceType taxDeviceType) {
        String type = calculationConfig.get(taxDeviceType.getCode());
        BigDecimal diff = (item.getAmountWithoutTax().multiply(item.getTaxRate()).subtract(item.getTaxAmount())).subtract(item.getDiscountWithoutTax().multiply(item.getTaxRate()).subtract(item.getDiscountTax()));
        return ErrorAmountPolicyEnum.ABS.name().equals(type) ?
                (diff.abs()) :
                diff;
    }

    /**
     * 计算票面总误差
     * @param totalErrorAmount
     * @param currentItemAmount
     * @return
     */
    protected Boolean totalErrorAmountCalculation(BigDecimal totalErrorAmount, BigDecimal currentItemAmount, BigDecimal basicInvoiceMaxErrorAmount, TaxDeviceType taxDeviceType) {
        /**
         * sum((A-B).abs).half_down  百旺 UKEY 拆票规则下，不再校验票面总尾差
         */
        if (TaxDeviceType.isBW(taxDeviceType)) {
            return Boolean.FALSE;
        }
        /**
         * SUM(A-B）.abs，
         */
        else {
            return totalErrorAmount.add(currentItemAmount).abs().compareTo(basicInvoiceMaxErrorAmount) > 0;
        }
    }

}
