package com.xforceplus.phoenix.split.service.dataflow.impl.invoiceLimit;

import com.xforceplus.phoenix.split.config.SplitProperties;
import com.xforceplus.phoenix.split.constant.ErrorAmountPolicyEnum;
import com.xforceplus.phoenix.split.constant.TaxDeviceType;
import com.xforceplus.phoenix.split.model.BillItem;
import com.xforceplus.phoenix.split.model.SplitRule;
import lombok.RequiredArgsConstructor;

import java.math.BigDecimal;
import java.util.Optional;

@RequiredArgsConstructor
public abstract class AbstractCombineItemService {

    private final SplitProperties splitProperties;

    /**
     * 明细实际金额
     * 旧逻辑，重构移动
     *
     * @param item                 单据明细
     * @param limitIsAmountWithTax 是否是含税限额
     * @return 明细实际金额
     */
    protected BigDecimal actualAmount(BillItem item, boolean limitIsAmountWithTax) {
        BigDecimal result;

        if (limitIsAmountWithTax) {
            result = item.getAmountWithTax().subtract(item.getDiscountWithTax());
        } else {
            result = item.getAmountWithoutTax().subtract(item.getDiscountWithoutTax());
        }

        return result;
    }

    /**
     * 返回明细生成预制发票时占用行数
     * 旧逻辑，重构移动
     *
     * @param item 单据明细
     * @return 明细生成预制发票时占用行数
     */
    protected int actualLineNum(BillItem item) {
        if (item.getDiscountWithoutTax().compareTo(BigDecimal.ZERO) > 0 && (item.getOutterDiscountWithoutTax().compareTo(BigDecimal.ZERO) > 0 || item.getOutterPrepayAmountWithoutTax().compareTo(BigDecimal.ZERO) > 0)) {
            return 2;
        }
        return 1;
    }

    /**
     * 单条明细计算尾差 考虑到百旺航信的误差调整
     * 旧逻辑，重构移动
     */
    protected BigDecimal errorAmountCalculation(BillItem item, TaxDeviceType taxDeviceType) {
        String type = splitProperties.getCalculationConfig().get(taxDeviceType.getCode());
        BigDecimal diff = (item.getAmountWithoutTax().multiply(item.getTaxRate()).subtract(item.getTaxAmount())).subtract(item.getDiscountWithoutTax().multiply(item.getTaxRate()).subtract(item.getDiscountTax()));
        return ErrorAmountPolicyEnum.ABS.name().equals(type) ? diff.abs() : diff;
    }

    /**
     * 计算票面总误差
     * 旧逻辑，重构移动
     */
    protected Boolean totalErrorAmountCalculation(BigDecimal totalErrorAmount, BigDecimal currentItemAmount, BigDecimal basicInvoiceMaxErrorAmount, TaxDeviceType taxDeviceType) {
        //sum((A-B).abs).half_down  百旺 UKEY 拆票规则下，不再校验票面总尾差
        if (TaxDeviceType.isBW(taxDeviceType)) {
            return Boolean.FALSE;
        }
        //SUM(A-B）.abs，
        else {
            return totalErrorAmount.add(currentItemAmount).abs().compareTo(basicInvoiceMaxErrorAmount) > 0;
        }
    }

    /**
     * 计算票面税额差是否超过业务配置数值
     * 旧逻辑，重构移动
     *
     * @param businessInvoiceMaxErrorAmount 业务配置的税额差，可能为空
     */
    public Boolean totalErrorAmountCalculationForBusiness(BigDecimal totalErrorAmount, BigDecimal currentItemErrorAmount, BigDecimal businessInvoiceMaxErrorAmount) {
        return businessInvoiceMaxErrorAmount != null &&
                businessInvoiceMaxErrorAmount.compareTo(BigDecimal.ZERO) != 0 &&
                totalErrorAmount.add(currentItemErrorAmount).abs().compareTo(businessInvoiceMaxErrorAmount) > 0;
    }

    /**
     * 旧逻辑，重构移动
     */
    protected BigDecimal getMaxErrorAmountByTaxDeviceType(TaxDeviceType taxDeviceType, SplitRule splitRule) {
        boolean isQdSpecialAddition = splitRule.isEnableAdvancedValidation();
        if (isQdSpecialAddition) {
            return BigDecimal.valueOf(1.27);
        }

        return Optional.ofNullable(splitProperties.getInvoiceMaxErrorAmountPolicy())
                .map(r -> r.get(taxDeviceType.getCode()))
                .orElse(new BigDecimal("1.26"));
    }
}
