package com.xforceplus.phoenix.split.service.impl;

import com.alibaba.fastjson.JSON;
import com.google.common.collect.Lists;
import com.xforceplus.phoenix.split.domain.ItemAmountInfo;
import com.xforceplus.phoenix.split.exception.SplitBizException;
import com.xforceplus.phoenix.split.model.BillItem;
import com.xforceplus.phoenix.split.model.SplitRule;
import com.xforceplus.phoenix.split.service.AmountSplitRuleUtils;
import com.xforceplus.phoenix.split.service.SplitRuleUtil;
import com.xforceplus.phoenix.split.util.ThreadLocalFactory;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.Arrays;
import java.util.List;
import java.util.function.BiConsumer;

import static java.math.RoundingMode.*;

public abstract class AbstractSplitBillItemAmountService {

    private static final BigDecimal MIN_QUANTITY = BigDecimal.ONE.movePointLeft(6);
    private static final BigDecimal refer = new BigDecimal("0.01");

    /**
     * 合并折扣金额
     */
    protected void mergeDiscountAmount(ItemAmountInfo itemAmountInfo) {
        itemAmountInfo.setAmountWithoutTax(itemAmountInfo.getAmountWithoutTax()
                .subtract(itemAmountInfo.getInnerDiscountWithoutTax())
                .subtract(itemAmountInfo.getInnerPrepayAmountWithoutTax())
        );

        itemAmountInfo.setAmountWithTax(itemAmountInfo.getAmountWithTax()
                .subtract(itemAmountInfo.getInnerDiscountWithTax())
                .subtract(itemAmountInfo.getInnerPrepayAmountWithTax())
        );

        itemAmountInfo.setTaxAmount(itemAmountInfo.getTaxAmount()
                .subtract(itemAmountInfo.getInnerDiscountTax())
                .subtract(itemAmountInfo.getInnerPrepayAmountTax())
        );

        itemAmountInfo.setDiscountWithoutTax(itemAmountInfo.getOutterDiscountWithoutTax()
                .add(itemAmountInfo.getOutterPrepayAmountWithoutTax())
        );

        itemAmountInfo.setDiscountWithTax(itemAmountInfo.getOutterDiscountWithTax()
                .add(itemAmountInfo.getOutterPrepayAmountWithTax())
        );

        itemAmountInfo.setDiscountTax(itemAmountInfo.getOutterDiscountTax()
                .add(itemAmountInfo.getOutterPrepayAmountTax())
        );
    }

    /**
     * 扣减拆分金额
     */
    protected void deductSplitItemAmount(SplitRule rule, ItemAmountInfo itemAmountInfo, ItemAmountInfo splitItemAmountInfo) {

        itemAmountInfo.deductNewItemAmount(splitItemAmountInfo);

        if (AmountSplitRuleUtils.isUnitPriceAndQuantityInteger(rule.getAmountSplitRule()) ||
                AmountSplitRuleUtils.recalculateQuantityRounding(rule.getAmountSplitRule()) ||
                AmountSplitRuleUtils.isUnitPrice(rule.getAmountSplitRule()) ||
                AmountSplitRuleUtils.isUnitPriceAndRecalculateQuantity(rule.getAmountSplitRule())) {

            itemAmountInfo.setQuantity(itemAmountInfo.getQuantity().subtract(splitItemAmountInfo.getQuantity()));
        }
    }

    /**
     * 处理最后一条
     *
     * @param itemAmountInfo
     * @param hasQuantity
     * @param rule
     */
    protected void processLastItemUnitPriceAndQuantity(ItemAmountInfo itemAmountInfo, boolean hasQuantity, SplitRule rule) {
        if (!hasQuantity) {
            return;
        }

        BigDecimal quantity = itemAmountInfo.getQuantity();
        BigDecimal price = itemAmountInfo.getUnitPrice();
        if (quantity.compareTo(BigDecimal.ZERO) == 0) {
            quantity = MIN_QUANTITY;
        }

        /**
         * 当前的数量*单价 四舍五入 = 不含税金额，直接返回
         */
        if (checkAmount(quantity, price, itemAmountInfo.getAmountWithoutTax())) {
            return;
        }
        /**
         * 如果允许反算数量
         */
        if (AmountSplitRuleUtils.isUnitPriceAndRecalculateQuantity(rule.getAmountSplitRule())) {
            quantity = itemAmountInfo.getAmountWithoutTax().divide(price, 6, DOWN);
            if (checkAmount(quantity, price, itemAmountInfo.getAmountWithoutTax())) {
                itemAmountInfo.setQuantity(quantity);
                return;
            }
            quantity = itemAmountInfo.getAmountWithoutTax().divide(price, 6, UP);
            if (checkAmount(quantity, price, itemAmountInfo.getAmountWithoutTax())) {
                itemAmountInfo.setQuantity(quantity);
                return;
            }
        }
        /**
         * 如果不允许反算数量，直接反算单价 默认，如果反算单价无法保证需要处理 TODO
         */
        itemAmountInfo.setUnitPrice(itemAmountInfo.getAmountWithoutTax()
                .divide(quantity, rule.getUnitPriceScale(), HALF_UP));
    }

    /**
     * 判断单价 数量与金额匹配问题
     * @param quantity
     * @param price
     * @param amountWithOutTax
     * @return
     */
    protected Boolean checkAmount(BigDecimal quantity, BigDecimal price, BigDecimal amountWithOutTax) {
        BigDecimal currentAmount = quantity.multiply(price).setScale(3, HALF_UP);
        if ((amountWithOutTax.subtract(currentAmount)).abs().compareTo(refer) < 0) {
            return Boolean.TRUE;
        }
        return Boolean.FALSE;
    }

    protected List<ItemAmountInfo> processLastItemErrorAmount(ItemAmountInfo originalItem, List<ItemAmountInfo> result, SplitRule rule) {

        if (originalItem == null) {
            return result;
        }

        /**
         * 拆分算法优先计算金额
         * 最后一条明细的折扣金额可能已经为0，需要从后往前找存在误差的折扣金额
         */
        BigDecimal taxAmount = originalItem.getTaxAmount();
        BigDecimal expectedTaxAmount =
                (originalItem.getAmountWithoutTax().subtract(originalItem.getDeductions()))
                        .multiply(originalItem.getTaxRate());
        BigDecimal errorAmount = taxAmount.subtract(expectedTaxAmount);

        ItemAmountInfo discountItem = findEffectiveDiscountItem(originalItem, result);
        BigDecimal discountTax = discountItem.getDiscountTax();
        BigDecimal expectedDiscountTax = discountItem.getDiscountWithoutTax().multiply(discountItem.getTaxRate());
        BigDecimal errorDiscountTax = discountTax.subtract(expectedDiscountTax);

        BillItem billItem = getBillItemFromThreadLocal();
        BigDecimal offsetAmount = SplitRuleUtil.getErrorTaxAmount(rule, billItem.isRedItem());

        /**
         * 计算什么样的场景下是肯定不用调整尾差的
         * v1:(原误差 + 折扣误差).abs <= refer
         * v2:(原误差 - 折扣误差).abs <= refer
         * v3:最后一条正常明细误差小于0.01 && 多条折扣明细误差小于0.01
         *
         */
        if (errorAmount.abs().compareTo(offsetAmount) <= 0 &&
                errorDiscountTax.abs().compareTo(offsetAmount) <= 0) {

            result.add(originalItem);
            return result;
        }

        /**
         * 调整税额尾差
         * 计算调整范围
         * 如果税额大于0.01，计算调整条数
         * 如果折扣税额大于0.01 计算调整折扣税额
         *
         * 如果两个都不大于0.01 ，只调整税额
         */
        if (errorAmount.abs().compareTo(offsetAmount) > 0) {
            this.doAdjustNew(originalItem, errorAmount, offsetAmount, result,
                    (x, y) -> {
                        x.setTaxAmount(x.getTaxAmount().add(y));
                        x.setAmountWithTax(x.getTaxAmount().add(x.getAmountWithoutTax()));
                    },
                    (x, y) -> {
                        x.setTaxAmount(x.getTaxAmount().add(y));
                        x.setAmountWithTax(x.getAmountWithoutTax().add(x.getTaxAmount()));
                    });
        }

        /**
         * 调整折扣税额尾差
         */
        if (errorDiscountTax.abs().compareTo(offsetAmount) > 0) {
            result = Lists.reverse(result);

            this.doAdjustNew(discountItem, errorDiscountTax, offsetAmount, result,
                    (x, y) -> {
                        x.setDiscountTax(x.getDiscountTax().add(y));
                        x.setDiscountWithTax(x.getDiscountTax().add(x.getDiscountWithoutTax()));
                    },
                    (x, y) -> {
                        x.setDiscountTax(x.getDiscountTax().add(y));
                        x.setDiscountWithTax(x.getDiscountWithoutTax().add(x.getDiscountTax()));
                    });
        }

        result.add(originalItem);
        return result;
    }

    private BillItem getBillItemFromThreadLocal() {
        ThreadLocal<Object> threadLocal = ThreadLocalFactory.fetchThreadLocal();
        Object object = threadLocal.get();
        BillItem threadLocalItem = object == null ? new BillItem() : (BillItem) object;
        return JSON.parseObject(JSON.toJSONString(threadLocalItem), BillItem.class);
    }

    /**
     * 从后往前寻找折扣不为0的明细
     *
     * @param originalItem 最后一条明细
     * @param result       组装完成的明细
     * @return
     */
    protected ItemAmountInfo findEffectiveDiscountItem(ItemAmountInfo originalItem, List<ItemAmountInfo> result) {

        BigDecimal discountTax = originalItem.getDiscountTax();

        if (BigDecimal.ZERO.compareTo(discountTax) == 0) {

            for (int i = result.size() - 1; i >= 0; i--) {
                ItemAmountInfo item = result.get(i);
                if (item.getDiscountTax().compareTo(BigDecimal.ZERO) != 0) {
                    return item;
                }
            }

            return originalItem;

        } else {

            return originalItem;

        }
    }

    private List<ItemAmountInfo> doAdjustNew(
            final ItemAmountInfo splitAmountInfo,
            final BigDecimal errorAmount,
            final BigDecimal offsetAmount,
            final List<ItemAmountInfo> result,
            final BiConsumer<ItemAmountInfo, BigDecimal> consumer,
            final BiConsumer<ItemAmountInfo, BigDecimal> consumer2
    ) {
        /**
         * 尾差出现超过n分的情况 开始分摊
         * offsetAmount：1分、或  6分钱
         */
        int adjustNum;
        BigDecimal remainder;
        if (errorAmount.abs().compareTo(offsetAmount) > 0) {
            BigDecimal divideResult = errorAmount.divide(offsetAmount, 4);
            adjustNum = divideResult.setScale(0, DOWN).abs().intValue();
            remainder = errorAmount.abs().subtract(offsetAmount.abs().multiply(BigDecimal.valueOf(adjustNum))).setScale(2, DOWN);

            //有余数
            if (remainder.signum() > 0) {
                adjustNum = adjustNum + 1;
            }
        } else {
            adjustNum = 1;
            remainder = errorAmount.abs();
        }

        BigDecimal[] array = new BigDecimal[adjustNum];
        Arrays.fill(array, offsetAmount);
        if (remainder.signum() > 0) {
            array[adjustNum - 1] = remainder;
        }

        /**
         * 本质原因是业务单明细行税差未校验
         * 临时解决方案：
         * 判断adjustNum和result.size 如果大于，则直接返回业务异常
         * 拆票插件方法中没有单据信息
         */
        if (!CollectionUtils.isEmpty(result) && adjustNum > result.size()) {

            ThreadLocal<Object> threadLocal = ThreadLocalFactory.fetchThreadLocal();

            Object object = threadLocal.get();

            BillItem threadLocalItem = object == null ? new BillItem() : (BillItem) object;

            BillItem billItem = JSON.parseObject(JSON.toJSONString(threadLocalItem), BillItem.class);

            threadLocal.remove();

            throw new SplitBizException(String.format("明细id = [%s]  名称 [%s] 不含税金额[%s]*税率[%s] 与 税额[%s] 之间误差超过" + offsetAmount.toPlainString(),
                    billItem.getSalesbillItemId(), billItem.getItemName(),
                    billItem.getAmountWithoutTax(), billItem.getTaxRate(), billItem.getTaxAmount()));

        }
        BigDecimal adjustTotalAmount = Arrays.stream(array).reduce(BigDecimal.ZERO, BigDecimal::add);
        adjustTotalAmount = errorAmount.compareTo(BigDecimal.ZERO) > 0 ? adjustTotalAmount : BigDecimal.ZERO.subtract(adjustTotalAmount);
        consumer2.accept(splitAmountInfo, BigDecimal.ZERO.subtract(adjustTotalAmount));
        return adjustResNew(array, result, consumer);
    }


    /**
     * 从前面的明细中 调整进位退位，保证尾差正确
     *
     * @param adjustValues
     * @param result
     * @param consumer
     * @return
     */
    private List<ItemAmountInfo> adjustResNew(
            BigDecimal[] adjustValues,
            List<ItemAmountInfo> result,
            BiConsumer<ItemAmountInfo, BigDecimal> consumer
    ) {
        for (int index = adjustValues.length - 1; index >= 0; index--) {
            consumer.accept(result.get(index), adjustValues[index]);
        }
        return result;
    }

}
