package com.xforceplus.phoenix.split.service.impl;

import com.alibaba.fastjson.JSON;
import com.google.common.base.Joiner;
import com.google.common.collect.Lists;
import com.xforceplus.phoenix.split.constant.InvoiceType;
import com.xforceplus.phoenix.split.constant.PreInvoiceTemplateVersionStatus;
import com.xforceplus.phoenix.split.constant.SaleListOptionEnum;
import com.xforceplus.phoenix.split.constant.TaxInvoiceSourceEnum;
import com.xforceplus.phoenix.split.exception.SplitBizException;
import com.xforceplus.phoenix.split.model.BillInfo;
import com.xforceplus.phoenix.split.model.BillItem;
import com.xforceplus.phoenix.split.model.ItemTypeCodeEnum;
import com.xforceplus.phoenix.split.model.PreInvoiceItem;
import com.xforceplus.phoenix.split.model.PreInvoiceMain;
import com.xforceplus.phoenix.split.model.SplitPreInvoiceInfo;
import com.xforceplus.phoenix.split.model.SplitRule;
import com.xforceplus.phoenix.split.model.ZeroTaxOption;
import com.xforceplus.phoenix.split.service.PreInvoiceGenerateService;
import com.xforceplus.phoenix.split.service.RemarkService;
import com.xforceplus.phoenix.split.util.CommonTools;
import com.xforceplus.phoenix.split.util.ObjectCheckAndExcuteUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.nio.charset.Charset;
import java.util.*;
import java.util.stream.Collectors;

import static com.xforceplus.phoenix.split.constant.RemarkConstant.DEFAULT_SIZE;
import static com.xforceplus.phoenix.split.constant.RemarkConstant.ELECTRONIC_SIZE;
import static com.xforceplus.phoenix.split.constant.RemarkConstant.HBBM_SET;
import static com.xforceplus.phoenix.split.constant.RemarkConstant.LIST_GOODS_NAME;
import static com.xforceplus.phoenix.split.constant.RemarkConstant.QD_SIZE;
import static com.xforceplus.phoenix.split.constant.RemarkConstant.TAX_CODE_SHORT_NAME_JOINER_PREFIX;
import static com.xforceplus.phoenix.split.constant.RemarkConstant.TAX_CODE_SHORT_NAME_JOINER_SUFFIX;
import static com.xforceplus.phoenix.split.constant.SaleListDefaultRule.ALL_ELECTRONIC_SALE_LIST_OPTION_NUMBER;
import static java.math.BigDecimal.ROUND_HALF_UP;

/**
 * 创建预制发票抽象类
 */
@Service
public class PreInvoiceGenerateServiceImpl implements PreInvoiceGenerateService {

    private static final Logger logger = LoggerFactory.getLogger(PreInvoiceGenerateServiceImpl.class);

    public static final String CPY = "2";

    public static final String JDC = "jdc";

    public static final String CPY_ERROR_MESSAGE = "成品油发票数量，单位为必填";

    @Autowired
    protected RemarkService remarkService;

    @Value("${cargoNameLengthLimit:100}")
    private int cargoNameLengthLimit;

    @Value("${skCargoNameLengthLimit}")
    private int skCargoNameLengthLimit;

    @Value("${unit.replace.list}")
    private String replaceUnit;

    private static final Charset GBK = Charset.forName("GBK");

    /**
     * 生成预制发票
     * @param billInfo 单据信息
     * @param rule     规则
     * @return 预制发票list
     */
    @Override
    public List<SplitPreInvoiceInfo> createPreInvoice(BillInfo billInfo, SplitRule rule) {
        List<BillItem> billItemList = billInfo.getBillItems();
        if (CollectionUtils.isEmpty(billItemList)) {
            return Lists.newArrayList();
        }

        if (SaleListOptionEnum.ENABLE_LIST_OPTION.getValue().equals(rule.getSaleListOption()) ||
                SaleListOptionEnum.FORCE_LIST_OPTION.getValue().equals(rule.getSaleListOption())) {

            //单据已按成品油分组，所以判断任意一条明细即可判断本组是否为成品油
            if (billInfo.getInvoiceType().equals(InvoiceType.NORMAL_ROLL.value())) {
                throw new SplitBizException("卷票不支持销货清单");
            }
        }
        //生成预制发票
        SplitPreInvoiceInfo splitPreInvoiceInfo = generatePreInvoice(billInfo, rule);
        //追加备注
        splitPreInvoiceInfo.getPreInvoiceMain().setRemark(toLegalRemark(billInfo, rule));
        return Lists.newArrayList(splitPreInvoiceInfo);
    }

    /**
     * 生成预制发票 mei
     * @param billInfo 单据信
     * @param rule     拆票规则
     * @return 预制发票列表
     */
    protected SplitPreInvoiceInfo generatePreInvoice(BillInfo billInfo, SplitRule rule) {

        BillItem billItem0 = billInfo.getBillItems().get(0);
        //是否为红字业务单
        boolean isRedItem = billInfo.getBillItems().get(0).isRedItem();
        //零税率发票专改普
        if (rule.getZeroTaxOption().equals(ZeroTaxOption.PROCESS)) {
            if (billItem0.getTaxRate().equals(BigDecimal.ZERO)) {
                billInfo.setInvoiceType(InvoiceType.NORMAL.value());
            }
        }
        //特殊发票标记
        String specialInvoiceFlag = buildSpecialInvoiceFlag(billItem0.getItemTypeCode());

        SplitPreInvoiceInfo splitPreInvoiceInfo = new SplitPreInvoiceInfo();
        splitPreInvoiceInfo.setRuleId(String.valueOf(rule.getRuleId()));

        splitPreInvoiceInfo.setPreInvoiceMain(createPreInvoiceMain(billInfo, rule, specialInvoiceFlag));
        splitPreInvoiceInfo.setPreInvoiceItems(createPreInvoiceItems(billInfo.getBillItems(), rule, isRedItem, splitPreInvoiceInfo.getPreInvoiceMain().getInvoiceType()));
        splitPreInvoiceInfo.getPreInvoiceMain().setSaleListFileFlag(Byte.valueOf(getPrintSalesListFlag(splitPreInvoiceInfo, rule) ? "1" : "0"));

        if (splitPreInvoiceInfo.getPreInvoiceMain().getSaleListFileFlag().equals(Byte.valueOf("1"))) {
            if (!TaxInvoiceSourceEnum.QD.getValue().equals(rule.getTaxInvoiceSource()) &&
                    splitPreInvoiceInfo.getPreInvoiceItems().size() > rule.getSalesListMaxRow()) {
                throw new SplitBizException("明细行数超过销货清单行数！");
            }
        }

        splitPreInvoiceInfo.getPreInvoiceMain().setListGoodsName(Byte.valueOf("1").equals(splitPreInvoiceInfo.getPreInvoiceMain().getSaleListFileFlag()) ? LIST_GOODS_NAME : "");
        //设置主信息税率
        List<String> taxRateList = splitPreInvoiceInfo.getPreInvoiceItems().stream()
                .map(PreInvoiceItem::getTaxRate).distinct().map(BigDecimal::toPlainString).collect(Collectors.toList());
        splitPreInvoiceInfo.getPreInvoiceMain().setTaxRate(Joiner.on(",").skipNulls().join(taxRateList));

        assemblePreInvoiceMainAmount(splitPreInvoiceInfo);

        return splitPreInvoiceInfo;
    }

    /**
     * 组装票面信息金额
     * @param splitPreInvoiceInfo
     */
    protected void assemblePreInvoiceMainAmount(SplitPreInvoiceInfo splitPreInvoiceInfo) {

        /**
         * 根据明细算出三个金额
         */
        BigDecimal amountWithTax = splitPreInvoiceInfo.getPreInvoiceItems().stream()
                .map(PreInvoiceItem::getAmountWithTax).reduce(BigDecimal.ZERO, BigDecimal::add);
        BigDecimal discountWithTax = splitPreInvoiceInfo.getPreInvoiceItems().stream()
                .map(PreInvoiceItem::getDiscountWithTax).reduce(BigDecimal.ZERO, BigDecimal::add);

        BigDecimal newAmountWithTax = amountWithTax.subtract(discountWithTax).setScale(2, ROUND_HALF_UP);
        splitPreInvoiceInfo.getPreInvoiceMain().setAmountWithTax(newAmountWithTax);

        BigDecimal amountWithoutTax = splitPreInvoiceInfo.getPreInvoiceItems().stream()
                .map(PreInvoiceItem::getAmountWithoutTax).reduce(BigDecimal.ZERO, BigDecimal::add);
        BigDecimal discountWithoutTax = splitPreInvoiceInfo.getPreInvoiceItems().stream()
                .map(PreInvoiceItem::getDiscountWithoutTax).reduce(BigDecimal.ZERO, BigDecimal::add);

        BigDecimal newAmountWithoutTax = amountWithoutTax.subtract(discountWithoutTax).setScale(2, ROUND_HALF_UP);
        splitPreInvoiceInfo.getPreInvoiceMain().setAmountWithoutTax(newAmountWithoutTax);

        BigDecimal taxAmount = splitPreInvoiceInfo.getPreInvoiceItems().stream()
                .map(PreInvoiceItem::getTaxAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
        BigDecimal discountTax = splitPreInvoiceInfo.getPreInvoiceItems().stream()
                .map(PreInvoiceItem::getDiscountTax).reduce(BigDecimal.ZERO, BigDecimal::add);

        BigDecimal newTaxAmount = taxAmount.subtract(discountTax).setScale(2, ROUND_HALF_UP);
        splitPreInvoiceInfo.getPreInvoiceMain().setTaxAmount(newTaxAmount);

    }

    protected String buildSpecialInvoiceFlag(String itemTypeCode) {
        return ItemTypeCodeEnum.getEnumByValueIgnoreCase(itemTypeCode)
                .map(ItemTypeCodeEnum::getSpecialInvoiceFlag)
                .orElse("0");
    }

    /**
     * 全电场景下，是否打印销货清单 由 实际明细是否大于8判断
     * 是否打印销货清单
     * @param splitPreInvoiceInfo
     * @param rule
     * @return
     */
    protected boolean getPrintSalesListFlag(SplitPreInvoiceInfo splitPreInvoiceInfo, SplitRule rule) {

        int itemSize = 0;
        for (PreInvoiceItem preInvoiceItem : splitPreInvoiceInfo.getPreInvoiceItems()) {
            itemSize = hasDiscount(preInvoiceItem) ? itemSize + 2 : itemSize + 1;
        }

        //成品油不支持销货清单
        if (SaleListOptionEnum.DISABLE_LIST_OPTION.getValue().equals(rule.getSaleListOption()) ||
                CPY.equals(splitPreInvoiceInfo.getPreInvoiceMain().getSpecialInvoiceFlag())) {
            return false;
        }

        //红冲发票不拆票  如果行数超过发票限制 强制打印销货清单
        if (SaleListOptionEnum.FORCE_LIST_OPTION.getValue().equals(rule.getSaleListOption())) {
            return true;
        }

        /**
         * 红字蓝字关于是否打印销货清单逻辑一致
         * 税控设备 & 实际明细行大于规则配置的发票最大明细行
         * 全电类型 & 对接产线已切换最新版本 & 实际明细行大于8
         */
        return (TaxInvoiceSourceEnum.QD.getValue().equals(rule.getTaxInvoiceSource()) &&
                // rule.getAllElectronicProcessType() == 1 &&
                itemSize > ALL_ELECTRONIC_SALE_LIST_OPTION_NUMBER) ||
                itemSize > rule.getInvoiceItemMaxRow();
    }

    private boolean hasDiscount(PreInvoiceItem preInvoiceItem) {
        return preInvoiceItem.getDiscountWithoutTax() != null &&
                preInvoiceItem.getDiscountWithoutTax().compareTo(BigDecimal.ZERO) > 0;
    }

    /**
     * 生成预制发票主信息
     * @param billInfo           单据信息
     * @param rule               拆票规则
     * @param specialInvoiceFlag 特殊发票标记
     * @return 预制发票主信息
     */
    private PreInvoiceMain createPreInvoiceMain(BillInfo billInfo, SplitRule rule, String specialInvoiceFlag) {
        PreInvoiceMain preInvoiceMain = new PreInvoiceMain();
        BeanUtils.copyProperties(billInfo, preInvoiceMain);
        preInvoiceMain.setRuleId(rule.getRuleId());
        preInvoiceMain.setDisplayPriceQuality(Byte.valueOf(rule.getUnitPriceAmountOps()));
        preInvoiceMain.setSpecialInvoiceFlag(specialInvoiceFlag);
        preInvoiceMain.setTemplateVersion(
                null != rule.getInvoiceItemMaxRow() && rule.getInvoiceItemMaxRow() > 5 ?
                        PreInvoiceTemplateVersionStatus.EIGHT.getValue() : PreInvoiceTemplateVersionStatus.FIVE.getValue());
        preInvoiceMain.setTaxInvoiceSource(rule.getTaxInvoiceSource());
        return preInvoiceMain;
    }

    public static void negateBigDecimalField(Object object) {
        try {
            for (Field field : object.getClass().getDeclaredFields()) {
                if (field.getType() == BigDecimal.class) {
                    field.setAccessible(true);
                    BigDecimal bigDecimal = (BigDecimal) field.get(object);
                    if (Objects.nonNull(bigDecimal)) {
                        field.set(object, bigDecimal.negate());
                    }
                }
            }
        } catch (IllegalAccessException e) {
            logger.error("negateBigDecimalField error:{}", e);
            throw new RuntimeException(e);
        }
    }

    /**
     * 生成预制发票明细
     * @param itemList  单据明细列表
     * @param rule      拆票规则
     * @param isRedItem
     * @return 预制发票明细列表
     */
    private List<PreInvoiceItem> createPreInvoiceItems(List<BillItem> itemList, SplitRule rule, boolean isRedItem, final String invoiceType) {
        Map<String, String> unitMap =new HashMap<>();
        if(StringUtils.equals(rule.getTaxInvoiceSource(),TaxInvoiceSourceEnum.QD.getValue())) {
            unitMap = JSON.parseObject(replaceUnit, Map.class);
        }
        List<PreInvoiceItem> preInvoiceItems = new ArrayList<>(itemList.size());
        int cargoNameLengthLimit = this.getCargoNameLengthLimit(rule, invoiceType);
        for (BillItem billItem : itemList) {
            PreInvoiceItem preInvoiceItem = new PreInvoiceItem();
            preInvoiceItems.add(preInvoiceItem);
            BeanUtils.copyProperties(billItem, preInvoiceItem);
            billItem.setAmountWithoutTax(billItem.getAmountWithoutTax().setScale(2, RoundingMode.HALF_UP));
            billItem.setAmountWithTax(billItem.getAmountWithTax().setScale(2, RoundingMode.HALF_UP));
            billItem.setTaxAmount(billItem.getAmountWithTax().setScale(2, RoundingMode.HALF_UP));
            billItem.setDiscountWithoutTax(billItem.getDiscountWithoutTax().setScale(2, RoundingMode.HALF_UP));
            billItem.setDiscountWithTax(billItem.getDiscountWithTax().setScale(2, RoundingMode.HALF_UP));
            billItem.setDiscountTax(billItem.getDiscountTax().setScale(2, RoundingMode.HALF_UP));

            //红字业务单，金额设置为负数
            if (isRedItem) {
                negateBigDecimalField(preInvoiceItem);
            }
            preInvoiceItem.setUnitPrice(preInvoiceItem.getUnitPrice().abs());
            preInvoiceItem.setTaxRate(preInvoiceItem.getTaxRate().abs());
            preInvoiceItem.setCargoCode(billItem.getItemCode());
            setCargoName(billItem, preInvoiceItem, rule);
            if (StringUtils.equals(rule.getTaxInvoiceSource(), TaxInvoiceSourceEnum.QD.getValue()) && InvoiceType.isElectronic(invoiceType)) {
                if (this.cargoNameGtLimitForQd(preInvoiceItem.getCargoName(), cargoNameLengthLimit)) {
                    preInvoiceItem.setCargoName(StringUtils.substring(preInvoiceItem.getCargoName(), 0, cargoNameLengthLimit));
                }
            } else {
                if (this.cargoNameGtLimit(preInvoiceItem.getCargoName(), cargoNameLengthLimit)) {
                    preInvoiceItem.setCargoName(CommonTools.substring(preInvoiceItem.getCargoName(), cargoNameLengthLimit, GBK));
                }
            }
            preInvoiceItem.setPrintContentFlag(getPrintContentFlag(rule, billItem));
            preInvoiceItem.setTaxItem("");
            preInvoiceItem.setDeduction(billItem.getDeductions());
            preInvoiceItem.setGoodsTaxNo(getStrOrDefaultStr(billItem.getGoodsTaxNo(), ""));
            preInvoiceItem.setTaxPre(getStrOrDefaultStr(billItem.getTaxPre(), "0"));
            preInvoiceItem.setZeroTax(getStrOrDefaultStr(billItem.getZeroTax(), ""));
            preInvoiceItem.setPriceMethod(String.valueOf(rule.getPriceMethod().value()));
            ObjectCheckAndExcuteUtils.docheckAndExcute(rule, SplitRule::getShowSpecification, x -> x, x -> {
                preInvoiceItem.setItemSpec(StringUtils.EMPTY);
                return x;
            });

            emptyFieldValue(preInvoiceItem, rule);

            adjustBillItemAmountWithoutTax(preInvoiceItem);
            if(StringUtils.equals(rule.getTaxInvoiceSource(),TaxInvoiceSourceEnum.QD.getValue())&&unitMap!=null) {
               if(StringUtils.isNotBlank(preInvoiceItem.getQuantityUnit())) {
                   String replaceUnit = unitMap.get(preInvoiceItem.getQuantityUnit());
                   if(replaceUnit!=null) {
                       preInvoiceItem.setQuantityUnit(replaceUnit);
                   }
               }
            }
        }
        return preInvoiceItems;
    }

    /**
     * 全额折扣调整策略
     * 不含税金额等于折扣金额，税额和折扣税额相差1-2分
     */
    protected void adjustBillItemAmountWithoutTax(PreInvoiceItem item) {

        /**
         * 全额价外折扣
         */
        if (item.getDiscountWithTax().compareTo(BigDecimal.ZERO) != 0 &&
                item.getAmountWithoutTax().compareTo(item.getDiscountWithoutTax()) == 0) {

            BigDecimal offsetTaxAmount = item.getTaxAmount().subtract(item.getDiscountTax());
            if (offsetTaxAmount.compareTo(BigDecimal.valueOf(0.01)) == 0 ||
                    offsetTaxAmount.compareTo(BigDecimal.valueOf(0.02)) == 0) {

                item.setDiscountTax(item.getDiscountTax().add(BigDecimal.valueOf(0.01)));
                item.setDiscountWithoutTax(item.getDiscountWithoutTax().subtract(BigDecimal.valueOf(0.01)));

            }
        }

        /**
         * 全额价内折扣
         * 包含正数0.01和负数0.01场景
         */
        if (item.getAmountWithTax().compareTo(BigDecimal.valueOf(0.01)) == 0 &&
                item.getTaxAmount().compareTo(BigDecimal.valueOf(0.01)) == 0 &&
                item.getDiscountWithTax().compareTo(BigDecimal.ZERO) == 0) {

            item.setAmountWithoutTax(BigDecimal.valueOf(0.01));
            item.setTaxAmount(BigDecimal.ZERO);

        } else if (item.getAmountWithTax().compareTo(BigDecimal.valueOf(-0.01)) == 0 &&
                item.getTaxAmount().compareTo(BigDecimal.valueOf(-0.01)) == 0 &&
                item.getDiscountWithTax().compareTo(BigDecimal.ZERO) == 0) {

            item.setAmountWithoutTax(BigDecimal.valueOf(-0.01));
            item.setTaxAmount(BigDecimal.ZERO);
        }
    }

    protected void emptyFieldValue(PreInvoiceItem preInvoiceItem, SplitRule rule) {

        String itemTypeCode = preInvoiceItem.getItemTypeCode();

        BigDecimal amountWithoutTax = preInvoiceItem.getAmountWithoutTax();
        boolean isRedItem = amountWithoutTax != null && amountWithoutTax.compareTo(BigDecimal.ZERO) < 0;

        /**
         * 成品油或者矿产品蓝票必须打印数量单价，可无视规则
         */
        if (Objects.equals("1", rule.getUnitPriceAmountOps()) &&
                !((ItemTypeCodeEnum.OIL.getValue().equals(itemTypeCode) || ItemTypeCodeEnum.MINERALS.getValue().equals(itemTypeCode)) &&
                        !isRedItem)) {

            preInvoiceItem.setUnitPrice(BigDecimal.ZERO);
            preInvoiceItem.setQuantity(BigDecimal.ZERO);
        }

        if (!rule.isPrintItemSpecFlag()) {
            preInvoiceItem.setItemSpec("");
        }
    }

    private String getStrOrDefaultStr(String str, String defaultStr) {
        if (StringUtils.isEmpty(str)) {
            return defaultStr;
        }
        return str;
    }

    private String getPrintContentFlag(SplitRule rule, BillItem billItem) {
        return BigDecimal.ZERO.compareTo(billItem.getQuantity()) == 0 && BigDecimal.ZERO.compareTo(billItem.getUnitPrice()) == 0
                ? "1" : StringUtils.isNotEmpty(rule.getUnitPriceAmountOps()) ? rule.getUnitPriceAmountOps() : "0";
    }

    private void setCargoName(BillItem billItem, PreInvoiceItem preInvoiceItem, SplitRule rule) {

        boolean hideCargoNameItemShortName = rule.isHideCargoNameItemShortName();

        if (StringUtils.isNotBlank(billItem.getItemShortName()) &&
                !hideCargoNameItemShortName) {

            String cargoName = TAX_CODE_SHORT_NAME_JOINER_PREFIX + billItem.getItemShortName()
                    + TAX_CODE_SHORT_NAME_JOINER_SUFFIX;

            if (StringUtils.isNotBlank(billItem.getJoinName())) {
                cargoName = cargoName + billItem.getJoinName();
            } else {
                cargoName = cargoName + billItem.getItemName();
            }
            preInvoiceItem.setCargoName(cargoName);

        } else {

            if (StringUtils.isNotBlank(billItem.getJoinName())) {
                preInvoiceItem.setCargoName(billItem.getJoinName());
            } else {
                preInvoiceItem.setCargoName(billItem.getItemName());
            }
        }
    }

    private int getCargoNameLengthLimit(SplitRule rule, String invoiceType) {
        if (StringUtils.equals(rule.getTaxInvoiceSource(), TaxInvoiceSourceEnum.QD.getValue())
                && InvoiceType.isElectronic(invoiceType)) {
            return Optional.ofNullable(rule.getCargoNameLength())
                    .filter(r -> r <= this.cargoNameLengthLimit)
                    .orElse(this.cargoNameLengthLimit);
        } else {
            return Optional.ofNullable(rule.getCargoNameLength())
                    .filter(r -> r <= this.skCargoNameLengthLimit)
                    .orElse(this.skCargoNameLengthLimit);
        }
    }

    private boolean cargoNameGtLimit(String cargoName, int cargoNameLengthLimit) {
        return cargoName.getBytes(GBK).length > cargoNameLengthLimit;
    }

    /**
     * 数电场景，明细商品名称长度校验
     *
     * @param cargoName            商品名称
     * @param cargoNameLengthLimit 商品名称长度限制
     * @return true：超过长度限制；false：长度限制内
     */
    private boolean cargoNameGtLimitForQd(String cargoName, int cargoNameLengthLimit) {
        return cargoName.length() > cargoNameLengthLimit;
    }

    /**
     * 追加备注
     * @param billInfo 单据信息
     * @param rule     拆票规则
     */
    private String toLegalRemark(BillInfo billInfo, SplitRule rule) {

        String remark = remarkService.splice(billInfo, rule);

        return adjustRemark(billInfo, rule, remark);

    }

    public String adjustRemark(BillInfo billInfo, SplitRule rule, String remark) {

        Integer remarkSizeLimit = rule.getCustomRemarkSize();

        if (Objects.isNull(remarkSizeLimit)) {

            if (TaxInvoiceSourceEnum.QD.getValue().equals(rule.getTaxInvoiceSource())) {
                remarkSizeLimit = QD_SIZE;
            } else {
                if (InvoiceType.ELECTRONIC.value().equals(billInfo.getInvoiceType())) {
                    remarkSizeLimit = ELECTRONIC_SIZE;
                } else {
                    remarkSizeLimit = DEFAULT_SIZE;
                }
            }

        }

        if (TaxInvoiceSourceEnum.QD.getValue().equals(rule.getTaxInvoiceSource()) && InvoiceType.isElectronic(billInfo.getInvoiceType())) {
            int endIndex = remarkSizeLimit > remark.length() ? remark.length() : remarkSizeLimit;
            return remark.substring(0, endIndex);
        } else {
            return RemarkService.toLegalRemark(remark, remarkSizeLimit);
        }
    }

}
