package com.xforceplus.phoenix.split.service.impl;

import com.xforceplus.phoenix.split.checker.SplitLimitChecker;
import com.xforceplus.phoenix.split.constant.SplitBillItemAmountType;
import com.xforceplus.phoenix.split.domain.ItemAmountInfo;
import com.xforceplus.phoenix.split.model.SplitRule;
import com.xforceplus.phoenix.split.service.AmountSplitRuleUtils;
import com.xforceplus.phoenix.split.service.SplitBillItemAmountService;
import com.xforceplus.phoenix.split.translator.mapper.SplitAmountMapper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.LinkedList;
import java.util.List;

/**
 * 根据数量限额，和金额限额一起拆分发票
 */
@Service
@Slf4j
public class SplitBillItemAmountByQuantityAndAmountLimitServiceImpl extends AbstractSplitBillItemAmountService implements SplitBillItemAmountService {

    private final SplitAmountMapper splitAmountMapper;
    private final SplitLimitChecker splitLimitChecker;

    public SplitBillItemAmountByQuantityAndAmountLimitServiceImpl(SplitAmountMapper splitAmountMapper,
                                                                  SplitLimitChecker splitLimitChecker) {
        super();

        this.splitAmountMapper = splitAmountMapper;
        this.splitLimitChecker = splitLimitChecker;
    }

    @Override
    public List<ItemAmountInfo> splitAmount(ItemAmountInfo itemAmountInfo, SplitRule rule) {
        ItemAmountInfo originalItem = splitAmountMapper.copy(itemAmountInfo);

        super.mergeDiscountAmount(originalItem);

        final SplitBillItemAmountType splitBillItemAmountType = this.chooseSplitBillItemAmountType(rule, originalItem);

        final BigDecimal quantityLimit = rule.getQuantityLimit();
        final BigDecimal invoiceLimit = rule.getInvoiceLimit();

        List<ItemAmountInfo> result = new LinkedList<>();
        while (splitLimitChecker.gtLimit(originalItem, rule)) {
            ItemAmountInfo splitItemAmountInfo;
            switch (splitBillItemAmountType) {
                case SPLIT_BY_QUANTITY:
                    splitItemAmountInfo = this.splitItemAmountInfoByQuantity(rule, originalItem, quantityLimit);
                    break;
                case SPLIT_BY_AMOUNT_WITH_TAX:
                    splitItemAmountInfo = this.splitItemAmountInfoByAmountWithTax(rule, originalItem, invoiceLimit);
                    break;
                case SPLIT_BY_AMOUNT_WITHOUT_TAX:
                default:
                    splitItemAmountInfo = this.splitItemAmountInfoByAmountWithoutTax(rule, originalItem, invoiceLimit);
                    break;
            }

            super.deductSplitItemAmount(rule, originalItem, splitItemAmountInfo);
            result.add(splitItemAmountInfo);
        }

        if (SplitBillItemAmountType.SPLIT_BY_QUANTITY != splitBillItemAmountType) {
            super.processLastItemUnitPriceAndQuantity(originalItem, true, rule);
        }

        result = super.processLastItemErrorAmount(originalItem, result, rule);

        return result;
    }

    /**
     * 根据数量限额拆分
     */
    protected ItemAmountInfo splitItemAmountInfoByQuantity(SplitRule rule, ItemAmountInfo originalItem, BigDecimal quantityLimit) {
        BigDecimal limitQuantity = this.formatLimitQuantity(rule, quantityLimit);
        return originalItem.createItemAmountByQuantity(limitQuantity);
    }

    /**
     * 根据不含税金额限额拆分
     */
    protected ItemAmountInfo splitItemAmountInfoByAmountWithoutTax(SplitRule rule, ItemAmountInfo originalItem, BigDecimal invoiceLimit) {
        return originalItem.createItemAmountByAmountWithoutTax(rule, invoiceLimit);
    }

    /**
     * 根据含税金额限额拆分
     */
    protected ItemAmountInfo splitItemAmountInfoByAmountWithTax(SplitRule rule, ItemAmountInfo originalItem, BigDecimal invoiceLimit) {
        return originalItem.createItemAmountByAmountWithTax(rule, invoiceLimit);
    }

    /**
     * 决定明细的限额拆分类型
     */
    private SplitBillItemAmountType chooseSplitBillItemAmountType(SplitRule rule, ItemAmountInfo originalItem) {
        final BigDecimal quantityLimit = rule.getQuantityLimit();
        final BigDecimal invoiceLimit = rule.getInvoiceLimit();

        final BigDecimal quantitySplitRate = quantityLimit.divide(originalItem.getQuantity(), 10, RoundingMode.DOWN);
        final BigDecimal amountSplitRate = rule.isLimitIsAmountWithTax()
                ? invoiceLimit.divide(originalItem.getAmountWithTax().subtract(originalItem.getDiscountWithTax()), 10, RoundingMode.DOWN)
                : invoiceLimit.divide(originalItem.getAmountWithoutTax().subtract(originalItem.getDiscountWithoutTax()), 10, RoundingMode.DOWN);

        if (quantitySplitRate.compareTo(amountSplitRate) <= 0 && quantityLimit.multiply(originalItem.getUnitPrice()).compareTo(invoiceLimit) <= 0) {
            return SplitBillItemAmountType.SPLIT_BY_QUANTITY;
        } else {
            return rule.isLimitIsAmountWithTax()
                    ? SplitBillItemAmountType.SPLIT_BY_AMOUNT_WITH_TAX
                    : SplitBillItemAmountType.SPLIT_BY_AMOUNT_WITHOUT_TAX;
        }
    }

    /**
     * 非最后一次拆分场景，格式化拆分数量
     */
    private BigDecimal formatLimitQuantity(SplitRule rule, BigDecimal currentQuantity) {
        if (AmountSplitRuleUtils.isUnitPriceAndQuantityInteger(rule.getAmountSplitRule())) {
            return currentQuantity.setScale(0, RoundingMode.DOWN);
        }
        return currentQuantity;
    }
}
