package com.xforceplus.phoenix.split.service;

import com.xforceplus.phoenix.split.checker.SplitLimitChecker;
import com.xforceplus.phoenix.split.constant.TaxInvoiceSourceEnum;
import com.xforceplus.phoenix.split.domain.ItemGroup;
import com.xforceplus.phoenix.split.exception.ItemUnitPriceException;
import com.xforceplus.phoenix.split.exception.SplitBizException;
import com.xforceplus.phoenix.split.model.BillItem;
import com.xforceplus.phoenix.split.model.PriceMethod;
import com.xforceplus.phoenix.split.model.SplitRule;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.List;

import static com.xforceplus.phoenix.split.constant.DataBasicRule.QD_QUANTITY_PRECISION;
import static com.xforceplus.phoenix.split.constant.DataBasicRule.QD_QUANTITY_SCALE;
import static com.xforceplus.phoenix.split.constant.DataBasicRule.QD_UNIT_PRICE_PRECISION;
import static com.xforceplus.phoenix.split.constant.DataBasicRule.QD_UNIT_PRICE_SCALE;
import static com.xforceplus.phoenix.split.constant.DataBasicRule.SK_UNIT_PRICE_PRECISION;
import static com.xforceplus.phoenix.split.constant.DataBasicRule.SK_UNIT_PRICE_SCALE;

/**
 * @Author chenlingwei
 * @create 6/1/22 4:15 PM
 */

@Service
@RequiredArgsConstructor
public class BasicDataProcessService {

    private final SplitLimitChecker splitLimitChecker;

    private static final BigDecimal DEFAULT_ALLOWABLE_ERROR = BigDecimal.valueOf(0.01);

    public void adjustUnitPrice(BillItem billItem, SplitRule rule) {

        adjustBusinessUnitPrice(billItem, rule);

        adjustBasicUnitPrice(billItem, rule);

    }

    /**
     * 20220418新增场景
     * 基础单价规则
     * 每行明细的单价小数位，整数+小数位不能超过20；超过则自动减去一位小数
     * 这条规则与业务规则无关
     * <p>
     * 20220720新增全电单价规则
     * 「单价」含小数点最多16位，小数位不超过13位
     * <p>
     * 20230222 单价小数位去除无用的0 兼容上游数据
     */
    public void adjustBasicUnitPrice(BillItem billItem, SplitRule rule) {

        BigDecimal unitPrice = billItem.getUnitPrice();
        BigDecimal quantity = billItem.getQuantity();
        if (unitPrice == null) {
            return;
        }

        unitPrice = unitPrice.stripTrailingZeros();
        /**
         * 发票无单价数量场景，不反算
         */
        if (BigDecimal.ZERO.compareTo(unitPrice) == 0 &&
                BigDecimal.ZERO.compareTo(quantity) == 0) {
            return;
        }

        int decimalNumber = unitPrice.scale();
        int precision = unitPrice.precision();
        int integerNumber = precision - decimalNumber;

        String taxInvoiceSource = rule.getTaxInvoiceSource();

        int precisionRuleScale = TaxInvoiceSourceEnum.QD.getValue().equals(taxInvoiceSource) ?
                QD_UNIT_PRICE_PRECISION : SK_UNIT_PRICE_PRECISION;

        int decimalRuleScale = TaxInvoiceSourceEnum.QD.getValue().equals(taxInvoiceSource) ?
                QD_UNIT_PRICE_SCALE : SK_UNIT_PRICE_SCALE;

        /**
         * 修改后的单价小数位 = (规则精度 - 实际整数位)与规则的小数精度取小
         */
        int actualUnitPriceScale = Math.min(precisionRuleScale - integerNumber, decimalRuleScale);

        if (precision > precisionRuleScale || decimalNumber > decimalRuleScale) {

            /**
             * 默认缩小小数位一位
             */
            BigDecimal newUnitPrice = calculationUnitPrice(billItem, actualUnitPriceScale, rule);
            BigDecimal newAmountWithoutTax = calculationAmountWithoutTax(newUnitPrice, quantity);
            BigDecimal newErrorAmount = calculationErrorAmount(newAmountWithoutTax, billItem);
            modifyUnitPriceOrProcessError(newErrorAmount, newUnitPrice, newAmountWithoutTax, billItem, rule,actualUnitPriceScale);
        }

    }

    /**
     * 20211214 BASE-49
     * 保单价前提下，明细行金额小数位大于规则小数位，按照规则重新反算小数位
     */
    public void adjustBusinessUnitPrice(BillItem billItem, SplitRule rule) {

        if (gtUnitPriceScale(billItem, rule.getUnitPriceScale())) {

            if (notNullQuantity(billItem) && !splitLimitChecker.gtLimit(billItem, rule)) {

                BigDecimal newUnitPrice = calculationUnitPrice(billItem, rule.getUnitPriceScale(), rule);
                BigDecimal newAmountWithoutTax = calculationAmountWithoutTax(newUnitPrice, billItem.getQuantity());
                BigDecimal newErrorAmount = calculationErrorAmount(newAmountWithoutTax, billItem);
                modifyUnitPriceOrProcessError(newErrorAmount, newUnitPrice, newAmountWithoutTax, billItem, rule,rule.getUnitPriceScale());

            }
        }
    }

    /**
     * 目前只针对全电，后续加入税控的数量限制
     * 20220720确认开票数量限制
     * 全电含小数点 16位 小数位13位
     * @param billItem
     * @param rule
     */
    public void adjustBasicQuantityPrecision(BillItem billItem, SplitRule rule) {

        BigDecimal quantity = billItem.getQuantity();
        if (quantity == null) {
            return;
        }
        quantity = quantity.stripTrailingZeros();
        BigDecimal unitPrice = billItem.getUnitPrice();

        int precisionRuleScale = QD_QUANTITY_PRECISION;
        int decimalRuleScale = QD_QUANTITY_SCALE;

        /**
         * 发票无单价数量场景，不反算
         */
        if (BigDecimal.ZERO.compareTo(unitPrice) == 0 &&
                BigDecimal.ZERO.compareTo(quantity) == 0) {
            return;
        }

        int decimalNumber = quantity.scale();
        int precision = quantity.precision();
        int integerNumber = precision - decimalNumber;

        int actualQuantityScale = Math.min((precisionRuleScale - integerNumber), decimalRuleScale);

        if (precision > precisionRuleScale || decimalNumber > decimalRuleScale) {

            BigDecimal newQuantity = calculationQuantity(billItem, actualQuantityScale);
            BigDecimal newAmountWithoutTax = calculationAmountWithoutTax(unitPrice, newQuantity);
            BigDecimal newErrorAmount = calculationErrorAmount(newAmountWithoutTax, billItem);
            modifyQuantityOrProcessError(newErrorAmount, newQuantity, newAmountWithoutTax, billItem, rule);
        }

    }

    private BigDecimal calculationUnitPrice(BillItem billItem, Integer unitPriceScale, SplitRule rule) {

        BigDecimal originUnitPrice = billItem.getUnitPrice();
        // boolean isRedItem = billItem.getAmountWithoutTax().compareTo(BigDecimal.ZERO) < 0;
        boolean isRedItem = billItem.getAmountWithoutTax().compareTo(BigDecimal.ZERO) < 0 || billItem.isRedItem();

        /**
         * 整数不做处理
         */
        if (new BigDecimal(originUnitPrice.intValue()).compareTo(originUnitPrice) == 0) {
            return originUnitPrice.setScale(unitPriceScale);
        }

        /**
         * 兼容sk下含税单子 不含税限额拆分情况
         */
        if (TaxInvoiceSourceEnum.QD.getValue().equals(rule.getTaxInvoiceSource()) &&
                PriceMethod.WITH_TAX == rule.getPriceMethod() && !isRedItem) {

            BigDecimal amount = billItem.getAmountWithTax()
                    .subtract(billItem.getInnerDiscountWithTax())
                    .subtract(billItem.getInnerPrepayAmountWithTax());

            BigDecimal taxRate = billItem.getTaxRate();
            BigDecimal divideTaxRate = taxRate.add(BigDecimal.ONE);

            return amount.divide(divideTaxRate, 15, RoundingMode.HALF_UP).
                    divide(billItem.getQuantity(), unitPriceScale, RoundingMode.HALF_UP);

        } else {

            BigDecimal amount = billItem.getAmountWithoutTax()
                    .subtract(billItem.getInnerDiscountWithoutTax())
                    .subtract(billItem.getInnerPrepayAmountWithoutTax());

            return amount.divide(billItem.getQuantity(), unitPriceScale, RoundingMode.HALF_UP);
        }

    }

    private BigDecimal calculationQuantity(BillItem billItem, Integer quantityScale) {

        BigDecimal unitPrice = billItem.getUnitPrice();

        BigDecimal amount = billItem.getAmountWithoutTax()
                .subtract(billItem.getInnerDiscountWithoutTax())
                .subtract(billItem.getInnerPrepayAmountWithoutTax());
        /**
         * 新的单价计算规则 四舍五入更加合理
         */
        return amount.divide(unitPrice, quantityScale, RoundingMode.HALF_UP);
    }

    private boolean gtUnitPriceScale(BillItem billItem, Integer unitPriceScale) {
        return billItem.getUnitPrice() != null && billItem.getUnitPrice().scale() > unitPriceScale;
    }

    private boolean notNullQuantity(BillItem billItem) {
        return billItem.getQuantity() != null && billItem.getQuantity().compareTo(BigDecimal.ZERO) != 0;
    }

    private BigDecimal calculationAmountWithoutTax(BigDecimal newUnitPrice, BigDecimal quantity) {
        return newUnitPrice.multiply(quantity).setScale(2, RoundingMode.HALF_UP);
    }

    private BigDecimal calculationErrorAmount(BigDecimal newAmountWithoutTax, BillItem billItem) {
        BigDecimal amount = billItem.getAmountWithoutTax()
                .subtract(billItem.getInnerDiscountWithoutTax())
                .subtract(billItem.getInnerPrepayAmountWithoutTax());

        return newAmountWithoutTax.subtract(amount).abs();
    }

    /**
     * 设置新的单价
     * 1.新不含税金额跟原不含税金额相等则只设置新的单价
     * 2.不相等
     * 2.1 校验则直接报异常
     * 2.2 不校验则重算各个金额
     * @param errorAmount
     * @param newUnitPrice
     * @param newAmountWithoutTax
     * @param billItem
     * @param rule
     */
    private void modifyUnitPriceOrProcessError(BigDecimal errorAmount, BigDecimal newUnitPrice, BigDecimal newAmountWithoutTax, BillItem billItem, SplitRule rule,int scale) {
        if (errorAmount.compareTo(DEFAULT_ALLOWABLE_ERROR) >= 0) {
            if (rule.isIgnoreAllowableError()) {//不校验
                setNewAmount(newUnitPrice, newAmountWithoutTax, billItem);
            } else {
                if((TaxInvoiceSourceEnum.QD.getValue().equals(rule.getTaxInvoiceSource())&&PriceMethod.WITH_TAX==rule.getPriceMethod()))
                {
                    BigDecimal amount = billItem.getAmountWithoutTax()
                            .subtract(billItem.getInnerDiscountWithoutTax())
                            .subtract(billItem.getInnerPrepayAmountWithoutTax());
                    billItem.setUnitPrice(amount.divide(billItem.getQuantity(),scale,RoundingMode.HALF_UP));
                }
                if((TaxInvoiceSourceEnum.QD.getValue().equals(rule.getTaxInvoiceSource())&&PriceMethod.WITH_TAX!=rule.getPriceMethod()))
                {
                    throw new ItemUnitPriceException(String.format("明细id: [%s], 明细单号: [%s], 明细名称: [%s] 按指定小数位截取单价后超过允许的容差 ",
                            billItem.getSalesbillItemId(), billItem.getSalesbillItemNo(), billItem.getItemName()));
                }

            }
        } else {
            billItem.setUnitPrice(newUnitPrice);
        }

    }

    private void modifyQuantityOrProcessError(BigDecimal errorAmount, BigDecimal newQuantity, BigDecimal newAmountWithoutTax, BillItem billItem, SplitRule rule) {
        if (errorAmount.compareTo(DEFAULT_ALLOWABLE_ERROR) >= 0) {
            if (rule.isIgnoreAllowableError() ) {//不校验
                billItem.setQuantity(newQuantity);
            } else {
                if((TaxInvoiceSourceEnum.QD.getValue().equals(rule.getTaxInvoiceSource())&&PriceMethod.WITH_TAX!=rule.getPriceMethod())) {
//                    billItem.setUnitPrice(newQuantity);
                    throw new ItemUnitPriceException(String.format("明细id: [%s], 明细单号: [%s], 明细名称: [%s] 按指定小数位截取单价后超过允许的容差 ",
                            billItem.getSalesbillItemId(), billItem.getSalesbillItemNo(), billItem.getItemName()));
                }
            }
        } else {
            billItem.setQuantity(newQuantity);
        }

    }

    private void setNewAmount(BigDecimal newUnitPrice, BigDecimal newAmountWithoutTax, BillItem billItem) {
        billItem.setUnitPrice(newUnitPrice);
        billItem.setAmountWithoutTax(newAmountWithoutTax.add(billItem.getInnerPrepayAmountWithoutTax()).add(billItem.getInnerDiscountWithoutTax()));
        billItem.setTaxAmount(billItem.getAmountWithoutTax().multiply(billItem.getTaxRate()).setScale(2, RoundingMode.HALF_UP));
        billItem.setAmountWithTax(billItem.getAmountWithoutTax().add(billItem.getTaxAmount()));
    }

    /**
     * 预制发票全额折扣明细置换
     * @param itemGroups             完整的预制发票
     * @param fullDiscountItemGroups 全额折扣预制发票
     */
    public List<ItemGroup> adjustInvoiceItemOrder(List<ItemGroup> itemGroups, List<ItemGroup> fullDiscountItemGroups) {

        itemGroups.removeAll(fullDiscountItemGroups);
        if (CollectionUtils.isEmpty(itemGroups)) {
            throw new SplitBizException("发票金额为零,请检查拆票数据");
        }

        for (ItemGroup fullDiscountItemGroup : fullDiscountItemGroups) {

            /**
             * 全额折扣只需要置换一条
             */
            List<BillItem> fullDiscountBillItems = fullDiscountItemGroup.getBillItems();

            for (ItemGroup normalItemGroup : itemGroups) {

                List<BillItem> normalBillItems = normalItemGroup.getBillItems();
                int containFullDiscountCapacity = normalItemGroup.getContainFullDiscountCapacity();
                if (containFullDiscountCapacity > 0) {

                    /**
                     * 尾部明细替换
                     */
                    BillItem fullDiscountBillItem = fullDiscountBillItems.get(fullDiscountBillItems.size() - 1);
                    BillItem normalBillItem = normalBillItems.get(fullDiscountBillItems.size() - 1);

                    fullDiscountBillItems.remove(fullDiscountBillItem);
                    normalBillItems.remove(normalBillItem);

                    fullDiscountBillItems.add(normalBillItem);
                    normalBillItems.add(fullDiscountBillItem);

                    normalItemGroup.setContainFullDiscountCapacity(--containFullDiscountCapacity);
                    fullDiscountItemGroup.setDisplaced(true);
                    break;
                }

            }

            if (!fullDiscountItemGroup.isDisplaced()) {
                throw new SplitBizException("发票金额为零,请检查拆票数据");
            }
        }

        itemGroups.addAll(fullDiscountItemGroups);
        return itemGroups;
    }
}
