package com.xforceplus.phoenix.split.service.dataflow.impl;

import com.google.common.collect.Lists;
import com.xforceplus.phoenix.split.constant.InvoiceType;
import com.xforceplus.phoenix.split.constant.TaxInvoiceSourceEnum;
import com.xforceplus.phoenix.split.domain.ItemGroup;
import com.xforceplus.phoenix.split.model.BillItem;
import com.xforceplus.phoenix.split.model.ItemTypeCodeEnum;
import com.xforceplus.phoenix.split.model.SplitRule;
import com.xforceplus.phoenix.split.service.dataflow.AbstractBillItemGroupPlugin;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.List;

import static com.xforceplus.phoenix.split.model.ItemTypeCodeEnum.isSpecialInvoice;
import static com.xforceplus.phoenix.split.model.ItemTypeCodeEnum.isSpecialInvoiceAndJDC;

/**
 * 根据明细中的itemTypeCode分组
 * 逐步取代
 * @Author chenlingwei
 * @create 2023/1/30 3:24 PM
 * @see GroupItemBySplitCodePlugin
 */
@Service
public class GroupItemByItemTypeCodePlugin extends AbstractBillItemGroupPlugin {

    private static final String DEFAULT_ITEM_TYPE_CODE = "default_item_type_code";

    /**
     * 校验明细中是否有不能混拆的特殊票种
     * @param itemGroups
     * @return
     */
    @Override
    public boolean support(SplitRule rule, List<ItemGroup> itemGroups) {
        return itemGroups.stream().anyMatch(e -> {
            List<BillItem> billItems = e.getBillItems();
            if(TaxInvoiceSourceEnum.QD.getValue().equals(rule.getTaxInvoiceSource())){
                if(!CollectionUtils.isEmpty(billItems) && billItems.stream().anyMatch(e1 -> Lists.newArrayList(InvoiceType.SPECIAL.value(),InvoiceType.SE.value()).contains(rule.getInvoiceType()))){
                    return billItems.stream().anyMatch(e1 -> isSpecialInvoiceAndJDC(e1.getItemTypeCode()));
                }
            }

            return !CollectionUtils.isEmpty(billItems) && billItems.stream().anyMatch(e1 -> isSpecialInvoice(e1.getItemTypeCode()));
        });
    }

    /**
     * 特殊票种单独生成key
     * 非特殊票种或者错误的itemTypeCode都归位普通票种
     * @param billItem
     * @param rule
     * @return
     */
    @Override
    public String generateKey(BillItem billItem, SplitRule rule) {

        String itemTypeCode = billItem.getItemTypeCode();

        if (StringUtils.isEmpty(itemTypeCode)) {
            return DEFAULT_ITEM_TYPE_CODE;
        } else {
            ItemTypeCodeEnum itemTypeCodeEnum = ItemTypeCodeEnum.fromValue(itemTypeCode);
            if (itemTypeCodeEnum == null) {
                return DEFAULT_ITEM_TYPE_CODE;
            } else {
                return itemTypeCodeEnum.getValue();
            }
        }
    }

}
