package com.xforceplus.phoenix.split.service.dataflow.impl;

import com.alibaba.fastjson.JSON;
import com.google.common.base.Stopwatch;
import com.xforceplus.phoenix.split.checker.SplitLimitChecker;
import com.xforceplus.phoenix.split.constant.InvoiceItemOrder;
import com.xforceplus.phoenix.split.constant.InvoiceType;
import com.xforceplus.phoenix.split.constant.TaxDeviceType;
import com.xforceplus.phoenix.split.constant.TaxInvoiceSourceEnum;
import com.xforceplus.phoenix.split.domain.ItemAmountInfo;
import com.xforceplus.phoenix.split.domain.ItemGroup;
import com.xforceplus.phoenix.split.domain.RuleInfo;
import com.xforceplus.phoenix.split.exception.SplitRuleParamException;
import com.xforceplus.phoenix.split.model.BillInfo;
import com.xforceplus.phoenix.split.model.BillItem;
import com.xforceplus.phoenix.split.model.ItemTypeCodeEnum;
import com.xforceplus.phoenix.split.model.SplitRule;
import com.xforceplus.phoenix.split.service.SplitBillItemAmountServiceFactory;
import com.xforceplus.phoenix.split.service.dataflow.DataProcessPlugin;
import com.xforceplus.phoenix.split.util.ThreadLocalFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

/**
 * 顺序优先，张数优先 使用此插件
 */
@Service
public class SplitBillItemPlugin implements DataProcessPlugin {

    protected final Logger logger = LoggerFactory.getLogger(SplitBillItemPlugin.class);

    public static final String ORIGIN_INVOICE_CODE = "originInvoiceCode";

    public static final String ORIGIN_INVOICE_NO = "originInvoiceNo";

    public static final String RED_NOTIFICATION_NO = "redNotificationNo";

    @Autowired
    private SplitBillItemAmountServiceFactory splitBillItemAmountServiceFactory;
    @Autowired
    private SplitLimitChecker splitLimitChecker;

    @Override
    public List<ItemGroup> processData(List<ItemGroup> itemGroups, BillInfo billInfo, RuleInfo ruleInfo) {
        return processData(itemGroups, billInfo, ruleInfo, null);
    }

    @Override
    public List<ItemGroup> processData(List<ItemGroup> itemGroups, BillInfo billInfo, RuleInfo ruleInfo, TaxDeviceType taxDeviceType) {

        if (InvoiceItemOrder.ITEM_NO_MINIMUM_INVOICES.value().equals(ruleInfo.getSplitRule().getItemSort())) {
            return itemGroups;
        }

        Stopwatch stopwatch = Stopwatch.createStarted();
        final String invoiceType = billInfo.getInvoiceType();

        // long count = 0L;
        for (ItemGroup itemGroup : itemGroups) {
            List<BillItem> billItemList = new LinkedList<>();
            for (BillItem billItem : itemGroup.getBillItems()) {

                SplitRule rule = ruleInfo.getSplitRule();

                checkPriceAndQuantity(billItem);

                if (split(ruleInfo.getExtRule(), billItem, rule, invoiceType)) {
                    billItemList.addAll(splitBillItemByLimitAmount(billItem, rule, taxDeviceType));
                } else {
                    billItemList.add(billItem);
                }
            }

            itemGroup.setBillItems(billItemList);
        }
        logger.info("process bill items elapsed time = {} ms", stopwatch.elapsed().toMillis());

        return itemGroups;
    }

    protected void checkPriceAndQuantity(BillItem billItem) {
        BigDecimal price = billItem.getUnitPrice();
        BigDecimal quantity = billItem.getQuantity();
        if ((price.compareTo(BigDecimal.ZERO) != 0 && quantity.compareTo(BigDecimal.ZERO) == 0) ||
                (price.compareTo(BigDecimal.ZERO) == 0 && quantity.compareTo(BigDecimal.ZERO) != 0)) {
            throw new SplitRuleParamException(String.format("salesBillItemId = [%s] 单价跟数量必须同时为零或者同时不为零!", billItem.getSalesbillItemId()));
        }

    }

    /**
     * 拆票条件
     * 1.超过限额
     * 2.1蓝票都拆
     * 2.2红票且原号码代码红字信息编号都为空才拆
     *
     * @param extRule
     * @param billItem
     * @param rule
     * @param invoiceType
     * @return
     */
    protected boolean split(Map<String, Object> extRule, BillItem billItem, SplitRule rule, String invoiceType) {
        if (gtLimitAmount(billItem, rule, invoiceType)) {
            return !billItem.isRedItem() || splitRed(extRule, billItem);
        }
        return false;
    }

    private boolean splitRed(Map<String, Object> extRule, BillItem billItem) {
        return billItem.isRedItem() &&
                StringUtils.isEmpty(extRule.get(ORIGIN_INVOICE_CODE)) &&
                StringUtils.isEmpty(extRule.get(ORIGIN_INVOICE_NO)) &&
                StringUtils.isEmpty(extRule.get(RED_NOTIFICATION_NO));
    }

    private List<BillItem> splitBillItemByLimitAmount(BillItem billItem, SplitRule rule, TaxDeviceType taxDeviceType) {

        ThreadLocalFactory.initThreadLocal(billItem);

        ItemAmountInfo itemAmountInfo = copyBillItemAmountInfo(billItem, rule);

        List<ItemAmountInfo> itemAmountInfoList = splitBillItemAmountServiceFactory.getByRule(billItem, rule, taxDeviceType)
                        .splitAmount(itemAmountInfo, rule);

        ThreadLocalFactory.clearContext();

        logger.debug("split id = {} itemAmount, result = {}", billItem.getSalesbillItemId(), itemAmountInfoList);
        return reCreateBillItem(itemAmountInfoList, billItem);
    }

    protected List<BillItem> reCreateBillItem(List<ItemAmountInfo> itemAmountInfoList, BillItem billItem) {
        List<BillItem> result = new ArrayList<>(itemAmountInfoList.size());
        for (ItemAmountInfo itemAmountInfo : itemAmountInfoList) {
            BillItem newBillItem = createNewBillItem(itemAmountInfo, billItem);
            result.add(newBillItem);
        }

        return result;
    }

    private BillItem createNewBillItem(ItemAmountInfo itemAmountInfo, BillItem billItem) {
        BillItem result = JSON.parseObject(JSON.toJSONString(billItem), BillItem.class);

        result.setOriginalAmountWithoutTax(billItem.getAmountWithoutTax());

        result.setAmountWithoutTax(itemAmountInfo.getAmountWithoutTax());
        result.setAmountWithTax(itemAmountInfo.getAmountWithTax());
        result.setTaxAmount(itemAmountInfo.getTaxAmount());

        result.setDeductions(itemAmountInfo.getDeductions());

        result.setDiscountTax(itemAmountInfo.getDiscountTax());
        result.setDiscountWithoutTax(itemAmountInfo.getDiscountWithoutTax());
        result.setDiscountWithTax(itemAmountInfo.getDiscountWithTax());

        result.setUnitPrice(itemAmountInfo.getUnitPrice());
        result.setQuantity(itemAmountInfo.getQuantity());
        result.setSplitItem(true);

        return result;
    }

    protected ItemAmountInfo copyBillItemAmountInfo(BillItem billItem, SplitRule rule) {
        ItemAmountInfo itemAmountInfo = JSON.parseObject(JSON.toJSONString(billItem), ItemAmountInfo.class);
        itemAmountInfo.setPriceMethod(rule.getPriceMethod());
        return itemAmountInfo;
    }

    /**
     * 判断是否超限额
     * @param billItem 明细
     * @param rule     规则
     */
    protected boolean gtLimitAmount(BillItem billItem, SplitRule rule, String invoiceType) {

        String taxInvoiceSource = rule.getTaxInvoiceSource();
        String itemTypeCode = billItem.getItemTypeCode();
        boolean isAllElectronicSpecialInvoice = ItemTypeCodeEnum.isAllElectronicSpecialInvoice(itemTypeCode);
        if (TaxInvoiceSourceEnum.QD.getValue().equals(taxInvoiceSource) &&
                !InvoiceType.isAllElectronicNormalInvoice(invoiceType) &&
                isAllElectronicSpecialInvoice) {
            return false;
        }

        return splitLimitChecker.gtLimit(billItem, rule);
    }
}
