package com.xforceplus.phoenix.split.service.impl;

import com.alibaba.fastjson.JSON;
import com.google.common.collect.Lists;
import com.xforceplus.phoenix.split.domain.ItemAmountInfo;
import com.xforceplus.phoenix.split.exception.SplitBizException;
import com.xforceplus.phoenix.split.model.BillItem;
import com.xforceplus.phoenix.split.model.SplitRule;
import com.xforceplus.phoenix.split.service.AmountSplitRuleUtils;
import com.xforceplus.phoenix.split.service.SplitBillItemAmountService;
import com.xforceplus.phoenix.split.service.SplitRuleUtil;
import com.xforceplus.phoenix.split.util.ThreadLocalFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.function.BiConsumer;

import static java.math.BigDecimal.ROUND_DOWN;
import static java.math.BigDecimal.ROUND_HALF_UP;
import static java.math.RoundingMode.DOWN;
import static java.math.RoundingMode.HALF_UP;
import static java.math.RoundingMode.UP;

/**
 * 拆分超限额单据明细
 */
@Service
public class DefaultSplitBillItemAmountServiceImpl implements SplitBillItemAmountService {

    private Logger logger = LoggerFactory.getLogger(DefaultSplitBillItemAmountServiceImpl.class);

    public static final BigDecimal MIN_QUANTITY = BigDecimal.ONE.movePointLeft(6);

    @Override
    public List<ItemAmountInfo> splitAmount(ItemAmountInfo itemAmountInfo, SplitRule rule) {
        ItemAmountInfo originalItem = JSON.parseObject(JSON.toJSONString(itemAmountInfo), ItemAmountInfo.class);

        List<ItemAmountInfo> result = new LinkedList<>();
        boolean hasQuantity = itemAmountInfo.getQuantity().compareTo(BigDecimal.ZERO) != 0;
        mergeDiscountAmount(originalItem);
        BigDecimal limitAmount = rule.getInvoiceLimit();

        ItemAmountInfo splitItemAmountInfo = initFirstSplitItem(rule, originalItem, result, limitAmount);
        while (gtLimitAmount(originalItem, limitAmount)) {
            splitItemAmountInfo = copyFirstItemAmount(limitAmount, rule, originalItem, splitItemAmountInfo);
            deductSplitItemAmount(rule, originalItem, splitItemAmountInfo);
            result.add(splitItemAmountInfo);
        }

        processLastItemUnitPriceAndQuantity(originalItem, hasQuantity, rule);
        logger.debug("before process error amount result = {}", result);

        result = processLastItemErrorAmount(originalItem, result, rule);

        logger.debug("after process error amount result = {}", result);
        return result;
    }

    protected ItemAmountInfo initFirstSplitItem(SplitRule rule, ItemAmountInfo originalItem, List<ItemAmountInfo> result, BigDecimal limitAmount) {
        ItemAmountInfo splitItemAmountInfo = splitFirstItemAmountInfo(rule, originalItem, limitAmount);
        result.add(splitItemAmountInfo);
        return splitItemAmountInfo;
    }

    /**
     * 处理最后一条
     * @param itemAmountInfo
     * @param hasQuantity
     * @param rule
     */
    protected void processLastItemUnitPriceAndQuantity(ItemAmountInfo itemAmountInfo, boolean hasQuantity, SplitRule rule) {

        if (hasQuantity) {
            BigDecimal quantity = itemAmountInfo.getQuantity();
            BigDecimal price = itemAmountInfo.getUnitPrice();
            if (quantity.compareTo(BigDecimal.ZERO) == 0) {
                quantity = MIN_QUANTITY;
            }

            /**
             * 当前的数量*单价 四舍五入 = 不含税金额，直接返回
             */
            if (checkAmount(quantity, price, itemAmountInfo.getAmountWithoutTax())) {
                return;
            } else {
                /**
                 * 如果允许反算数量
                 */
                if (AmountSplitRuleUtils.isUnitPriceAndRecalculateQuantity(rule.getAmountSplitRule())) {
                    quantity = itemAmountInfo.getAmountWithoutTax().divide(price, 6, DOWN);
                    if (checkAmount(quantity, price, itemAmountInfo.getAmountWithoutTax())) {
                        itemAmountInfo.setQuantity(quantity);
                        return;
                    }
                    quantity = itemAmountInfo.getAmountWithoutTax().divide(price, 6, UP);
                    if (checkAmount(quantity, price, itemAmountInfo.getAmountWithoutTax())) {
                        itemAmountInfo.setQuantity(quantity);
                        return;
                    }
                }
                /**
                 * 如果不允许反算数量，直接反算单价 默认，如果反算单价无法保证需要处理 TODO
                 */
                itemAmountInfo.setUnitPrice(itemAmountInfo.getAmountWithoutTax()
                        .divide(quantity, rule.getUnitPriceScale(), HALF_UP));
            }

        }

    }

    /**
     * 判断单价 数量与金额匹配问题
     * @param quantity
     * @param price
     * @param amountWithOutTax
     * @return
     */
    protected Boolean checkAmount(BigDecimal quantity, BigDecimal price, BigDecimal amountWithOutTax) {
        BigDecimal currentAmount = quantity.multiply(price).setScale(3, HALF_UP);
        if ((amountWithOutTax.subtract(currentAmount)).abs().compareTo(refer) < 0) {
            return Boolean.TRUE;
        }
        return Boolean.FALSE;
    }

    /**
     * 扣减拆分金额
     * @param rule
     * @param itemAmountInfo
     * @param splitItemAmountInfo
     */
    protected void deductSplitItemAmount(SplitRule rule, ItemAmountInfo itemAmountInfo, ItemAmountInfo splitItemAmountInfo) {

        itemAmountInfo.deductNewItemAmount(splitItemAmountInfo);

        if (AmountSplitRuleUtils.isUnitPriceAndQuantityInteger(rule.getAmountSplitRule()) ||
                AmountSplitRuleUtils.recalculateQuantityRounding(rule.getAmountSplitRule()) ||
                AmountSplitRuleUtils.isUnitPrice(rule.getAmountSplitRule()) ||
                AmountSplitRuleUtils.isUnitPriceAndRecalculateQuantity(rule.getAmountSplitRule())) {

            itemAmountInfo.setQuantity(itemAmountInfo.getQuantity().subtract(splitItemAmountInfo.getQuantity()));
        }
    }

    /**
     * 复制拆分模版明细
     * @param limitAmount  限额
     * @param rule         拆票规则
     * @param originalItem 剩余的原明细信息
     * @param splitItem    拆分明细信息
     * @return
     */
    protected ItemAmountInfo copyFirstItemAmount(BigDecimal limitAmount, SplitRule rule,
            ItemAmountInfo originalItem, ItemAmountInfo splitItem) {

        ItemAmountInfo result = JSON.parseObject(JSON.toJSONString(splitItem), ItemAmountInfo.class);

        if (leftDiscountAmountLtSplitDiscountAmount(originalItem, result)) {
            reCalculateAmount(limitAmount, rule, originalItem, result);
        }
        return result;
    }

    /**
     * 剩余折扣已经小于预制发票折扣
     * @param limitAmount
     * @param rule
     * @param originalItemAmountInfo
     * @param result
     */
    protected void reCalculateAmount(BigDecimal limitAmount, SplitRule rule, ItemAmountInfo originalItemAmountInfo, ItemAmountInfo result) {

        result.setDiscountWithoutTax(originalItemAmountInfo.getDiscountWithoutTax());
        result.setDiscountTax(originalItemAmountInfo.getDiscountTax());
        result.setDiscountWithTax(originalItemAmountInfo.getDiscountWithTax());

        if (result.getAmountWithoutTax().subtract(result.getDiscountWithoutTax()).compareTo(limitAmount) > 0) {
            BigDecimal splitAmountWithoutTax = limitAmount.add(result.getDiscountWithoutTax()).setScale(2, ROUND_DOWN);
            BigDecimal splitTaxAmount = splitAmountWithoutTax.subtract(result.getDeductions())
                    .multiply(originalItemAmountInfo.getTaxRate())
                    .setScale(2, ROUND_DOWN);
            BigDecimal splitAmountWithTax = splitAmountWithoutTax.add(splitTaxAmount).setScale(2, ROUND_HALF_UP);

            result.setAmountWithTax(splitAmountWithTax);
            result.setAmountWithoutTax(splitAmountWithoutTax);
            result.setTaxAmount(splitTaxAmount);

            processUnitPriceOrQuantity(rule, result);
        }

    }

    /**
     * 判断剩余折扣是否小于拆分折扣
     */
    protected boolean leftDiscountAmountLtSplitDiscountAmount(ItemAmountInfo itemAmountInfo, ItemAmountInfo result) {
        return itemAmountInfo.getDiscountWithoutTax().compareTo(result.getDiscountWithoutTax()) <= 0;
    }

    /**
     * 按限额拆分金额
     * @return
     */
    protected ItemAmountInfo splitFirstItemAmountInfo(SplitRule rule, ItemAmountInfo originalItem, BigDecimal limitAmount) {
        ItemAmountInfo newItem = originalItem.createItemAmountByAmountWithoutTax(rule, limitAmount);
        deductSplitItemAmount(rule, originalItem, newItem);
        return newItem;
    }

    /**
     * 处理单价数量
     * @param rule
     * @param newItemAmountInfo
     */
    protected void processUnitPriceOrQuantity(SplitRule rule, ItemAmountInfo newItemAmountInfo) {
        if (AmountSplitRuleUtils.isUnitPrice(rule.getAmountSplitRule()) || AmountSplitRuleUtils.isUnitPriceAndRecalculateQuantity(rule.getAmountSplitRule())) {
            BigDecimal unitPrice = newItemAmountInfo.getUnitPrice();
            if (unitPrice != null && unitPrice.compareTo(BigDecimal.ZERO) > 0) {
                BigDecimal quantity = newItemAmountInfo.getAmountWithoutTax().divide(unitPrice, 6,
                        ROUND_HALF_UP);
                if (quantity.compareTo(BigDecimal.ZERO) == 0) {
                    quantity = MIN_QUANTITY;
                    newItemAmountInfo.setUnitPrice(newItemAmountInfo.getAmountWithoutTax().divide(quantity, rule.getUnitPriceScale(), ROUND_HALF_UP));
                }
                newItemAmountInfo.setQuantity(quantity);
            }
        } else if (AmountSplitRuleUtils.isUnitPriceAndQuantityInteger(rule.getAmountSplitRule()) ||
                AmountSplitRuleUtils.recalculateQuantityRounding(rule.getAmountSplitRule())) {
            BigDecimal quantity = newItemAmountInfo.getAmountWithoutTax().divide(newItemAmountInfo.getUnitPrice(), 0, ROUND_HALF_UP);
            newItemAmountInfo.setQuantity(quantity);

        } else if (AmountSplitRuleUtils.isQuantity(rule.getAmountSplitRule())) {
            if (newItemAmountInfo.getQuantity() != null && newItemAmountInfo.getQuantity().compareTo(BigDecimal.ZERO) > 0) {
                newItemAmountInfo.setUnitPrice(newItemAmountInfo.getAmountWithoutTax().divide(newItemAmountInfo.getQuantity(), rule.getUnitPriceScale(), ROUND_HALF_UP));
            }
        }
    }

    /**
     * 是否大于限额
     * @param itemAmountInfo
     * @param limitAmount
     * @return
     */
    protected boolean gtLimitAmount(ItemAmountInfo itemAmountInfo, BigDecimal limitAmount) {
        return itemAmountInfo.getAmountWithoutTax()
                .subtract(itemAmountInfo.getDiscountWithoutTax())
                .compareTo(limitAmount) > 0;
    }

    /**
     * 合并折扣金额
     * @param itemAmountInfo
     */
    protected void mergeDiscountAmount(ItemAmountInfo itemAmountInfo) {
        itemAmountInfo.setAmountWithoutTax(itemAmountInfo.getAmountWithoutTax()
                .subtract(itemAmountInfo.getInnerDiscountWithoutTax())
                .subtract(itemAmountInfo.getInnerPrepayAmountWithoutTax())
        );

        itemAmountInfo.setAmountWithTax(itemAmountInfo.getAmountWithTax()
                .subtract(itemAmountInfo.getInnerDiscountWithTax())
                .subtract(itemAmountInfo.getInnerPrepayAmountWithTax())
        );

        itemAmountInfo.setTaxAmount(itemAmountInfo.getTaxAmount()
                .subtract(itemAmountInfo.getInnerDiscountTax())
                .subtract(itemAmountInfo.getInnerPrepayAmountTax())
        );

        itemAmountInfo.setDiscountWithoutTax(itemAmountInfo.getOutterDiscountWithoutTax()
                .add(itemAmountInfo.getOutterPrepayAmountWithoutTax())
        );

        itemAmountInfo.setDiscountWithTax(itemAmountInfo.getOutterDiscountWithTax()
                .add(itemAmountInfo.getOutterPrepayAmountWithTax())
        );

        itemAmountInfo.setDiscountTax(itemAmountInfo.getOutterDiscountTax()
                .add(itemAmountInfo.getOutterPrepayAmountTax())
        );
    }

    private BillItem getBillItemFromThreadLocal() {
        ThreadLocal<Object> threadLocal = ThreadLocalFactory.fetchThreadLocal();
        Object object = threadLocal.get();
        BillItem threadLocalItem = object == null ? new BillItem() : (BillItem) object;
        return JSON.parseObject(JSON.toJSONString(threadLocalItem), BillItem.class);
    }

    /**
     * 处理最后一条金额尾差，以及最后多条的折扣尾差
     * @param originalItem
     * @param result
     * @return
     */
    protected List<ItemAmountInfo> processLastItemErrorAmount(ItemAmountInfo originalItem, List<ItemAmountInfo> result, SplitRule rule) {

        if (originalItem == null) {
            return result;
        }

        /**
         * 拆分算法优先计算金额
         * 最后一条明细的折扣金额可能已经为0，需要从后往前找存在误差的折扣金额
         */
        BigDecimal taxAmount = originalItem.getTaxAmount();
        BigDecimal expectedTaxAmount =
                (originalItem.getAmountWithoutTax().subtract(originalItem.getDeductions()))
                        .multiply(originalItem.getTaxRate());
        BigDecimal errorAmount = taxAmount.subtract(expectedTaxAmount);

        ItemAmountInfo discountItem = findEffectiveDiscountItem(originalItem, result);
        BigDecimal discountTax = discountItem.getDiscountTax();
        BigDecimal expectedDiscountTax = discountItem.getDiscountWithoutTax().multiply(discountItem.getTaxRate());
        BigDecimal errorDiscountTax = discountTax.subtract(expectedDiscountTax);

        BillItem billItem = getBillItemFromThreadLocal();
        BigDecimal offsetAmount = SplitRuleUtil.getErrorTaxAmount(rule, billItem.isRedItem());

        /**
         * 计算什么样的场景下是肯定不用调整尾差的
         * v1:(原误差 + 折扣误差).abs <= refer
         * v2:(原误差 - 折扣误差).abs <= refer
         * v3:最后一条正常明细误差小于0.01 && 多条折扣明细误差小于0.01
         *
         */
        if (errorAmount.abs().compareTo(offsetAmount) <= 0 &&
                errorDiscountTax.abs().compareTo(offsetAmount) <= 0) {

            result.add(originalItem);
            return result;
        }

        /**
         * 调整税额尾差
         * 计算调整范围
         * 如果税额大于0.01，计算调整条数
         * 如果折扣税额大于0.01 计算调整折扣税额
         *
         * 如果两个都不大于0.01 ，只调整税额
         */
        if (errorAmount.abs().compareTo(offsetAmount) > 0) {
            this.doAdjustNew(originalItem, errorAmount, offsetAmount, result,
                    (x, y) -> {
                        x.setTaxAmount(x.getTaxAmount().add(y));
                        x.setAmountWithTax(x.getTaxAmount().add(x.getAmountWithoutTax()));
                    },
                    (x, y) -> {
                        x.setTaxAmount(x.getTaxAmount().add(y));
                        x.setAmountWithTax(x.getAmountWithoutTax().add(x.getTaxAmount()));
                    });
        }

        /**
         * 调整折扣税额尾差
         */
        if (errorDiscountTax.abs().compareTo(offsetAmount) > 0) {
            result = Lists.reverse(result);

            this.doAdjustNew(discountItem, errorDiscountTax, offsetAmount, result,
                    (x, y) -> {
                        x.setDiscountTax(x.getDiscountTax().add(y));
                        x.setDiscountWithTax(x.getDiscountTax().add(x.getDiscountWithoutTax()));
                    },
                    (x, y) -> {
                        x.setDiscountTax(x.getDiscountTax().add(y));
                        x.setDiscountWithTax(x.getDiscountWithoutTax().add(x.getDiscountTax()));
                    });
        }

        result.add(originalItem);
        return result;
    }

    /**
     * 从后往前寻找折扣不为0的明细
     * @param originalItem 最后一条明细
     * @param result       组装完成的明细
     * @return
     */
    protected ItemAmountInfo findEffectiveDiscountItem(ItemAmountInfo originalItem, List<ItemAmountInfo> result) {

        BigDecimal discountTax = originalItem.getDiscountTax();

        if (BigDecimal.ZERO.compareTo(discountTax) == 0) {

            for (int i = result.size() - 1; i >= 0; i--) {
                ItemAmountInfo item = result.get(i);
                if (item.getDiscountTax().compareTo(BigDecimal.ZERO) != 0) {
                    return item;
                }
            }

            return originalItem;

        } else {

            return originalItem;

        }
    }

    private List<ItemAmountInfo> doAdjustNew(
            final ItemAmountInfo splitAmountInfo,
            final BigDecimal errorAmount,
            final BigDecimal offsetAmount,
            final List<ItemAmountInfo> result,
            final BiConsumer<ItemAmountInfo, BigDecimal> consumer,
            final BiConsumer<ItemAmountInfo, BigDecimal> consumer2
    ) {
        /**
         * 尾差出现超过n分的情况 开始分摊
         * offsetAmount：1分、或  6分钱
         */
        int adjustNum;
        BigDecimal remainder;
        if (errorAmount.abs().compareTo(offsetAmount) > 0) {
            BigDecimal divideResult = errorAmount.divide(offsetAmount, 4);
            adjustNum = divideResult.setScale(0, DOWN).abs().intValue();
            remainder = errorAmount.abs().subtract(offsetAmount.abs().multiply(BigDecimal.valueOf(adjustNum))).setScale(2, DOWN);

            //有余数
            if(remainder.signum() > 0) {
                adjustNum = adjustNum + 1;
            }
        } else {
            adjustNum = 1;
            remainder = errorAmount.abs();
        }

        BigDecimal[] array = new BigDecimal[adjustNum];
        Arrays.fill(array, offsetAmount);
        if(remainder.signum() > 0) {
            array[adjustNum-1] = remainder;
        }

        /**
         * 本质原因是业务单明细行税差未校验
         * 临时解决方案：
         * 判断adjustNum和result.size 如果大于，则直接返回业务异常
         * 拆票插件方法中没有单据信息
         */
        if (!CollectionUtils.isEmpty(result) && adjustNum > result.size()) {

            ThreadLocal<Object> threadLocal = ThreadLocalFactory.fetchThreadLocal();

            Object object = threadLocal.get();

            BillItem threadLocalItem = object == null ? new BillItem() : (BillItem) object;

            BillItem billItem = JSON.parseObject(JSON.toJSONString(threadLocalItem), BillItem.class);

            threadLocal.remove();

            throw new SplitBizException(String.format("明细id = [%s]  名称 [%s] 不含税金额[%s]*税率[%s] 与 税额[%s] 之间误差超过" + offsetAmount.toPlainString(),
                    billItem.getSalesbillItemId(), billItem.getItemName(),
                    billItem.getAmountWithoutTax(), billItem.getTaxRate(), billItem.getTaxAmount()));

        }
        BigDecimal adjustTotalAmount = Arrays.stream(array).reduce(BigDecimal.ZERO, BigDecimal::add);
        adjustTotalAmount = errorAmount.compareTo(BigDecimal.ZERO) > 0 ? adjustTotalAmount : BigDecimal.ZERO.subtract(adjustTotalAmount);
        consumer2.accept(splitAmountInfo,BigDecimal.ZERO.subtract(adjustTotalAmount));
        return adjustResNew(array, result, consumer);
    }


    /**
     * 从前面的明细中 调整进位退位，保证尾差正确
     * @param adjustValues
     * @param result
     * @param consumer
     * @return
     */
    private List<ItemAmountInfo> adjustResNew(
            BigDecimal[] adjustValues,
            List<ItemAmountInfo> result,
            BiConsumer<ItemAmountInfo, BigDecimal> consumer
    ) {
        for (int index = adjustValues.length - 1; index >= 0; index--) {
            consumer.accept(result.get(index), adjustValues[index]);
        }
        return result;
    }

}
