package com.xforceplus.phoenix.split.service.impl;

import com.google.common.base.Strings;
import com.xforceplus.phoenix.split.constant.InvoiceType;
import com.xforceplus.phoenix.split.constant.RemarkFieldNameEnum;
import com.xforceplus.phoenix.split.model.BillInfo;
import com.xforceplus.phoenix.split.model.BillItem;
import com.xforceplus.phoenix.split.model.FieldOfObj;
import com.xforceplus.phoenix.split.model.PriceMethod;
import com.xforceplus.phoenix.split.model.RemarkFieldMetadata;
import com.xforceplus.phoenix.split.model.SplitRule;
import com.xforceplus.phoenix.split.service.RemarkService;
import com.xforceplus.phoenix.split.util.CommonTools;
import com.xforceplus.phoenix.split.util.FieldUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.ibatis.reflection.ReflectionException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

import static com.xforceplus.phoenix.split.constant.RemarkConstant.*;

/**
 * 备注ServiceImpl
 */
@Service
public class RemarkServiceImpl implements RemarkService {

    private static final Logger logger = LoggerFactory.getLogger(RemarkServiceImpl.class);

    /**
     * 备注生成
     * @param billInfo 单据信息
     * @param rule     拆票规则
     * @return 完整备注
     */
    @Override
    public String splice(BillInfo billInfo, SplitRule rule) {
        if (InvoiceType.isVehicle(billInfo.getInvoiceType())) {
            return VEHICLE_REMARK;
        }

        /**
         * 备注拼接总规则
         * 1.拼接红字信息
         * 2.拼接拆票规则备注
         * 3.拼接扩展备注
         */
        StringBuilder sb = new StringBuilder();

        sb.append(handleRedInvoiceRemark(billInfo));

        List<RemarkFieldMetadata> remarkFieldMetadataBeanList = rule.getRemarkFiledMetadataBeanList();
        if (CollectionUtils.isNotEmpty(remarkFieldMetadataBeanList)) {

            /**
             * 1.处理原生remark
             * 2.处理拆票规则配置的remark
             */
            sb.append(this.handleRemark(billInfo, rule, remarkFieldMetadataBeanList));
        }

        appendSpace(sb);

        if (StringUtils.isNotBlank(rule.getExtRemark())) {
            sb.append(rule.getExtRemark());
        }
        return sb.toString();
    }

    /**
     * handleCustomRemarkV2
     * @param billInfo
     * @param rule
     * @param remarkFieldMetadataBeanList
     * @return
     */
    protected String handleRemark(BillInfo billInfo, SplitRule rule, List<RemarkFieldMetadata> remarkFieldMetadataBeanList) {
        List<BillItem> billItems = billInfo.getBillItems();

        StringBuilder sb = new StringBuilder();

        for (int i = 0; i < remarkFieldMetadataBeanList.size(); i++) {
            RemarkFieldMetadata metadataBean = remarkFieldMetadataBeanList.get(i);

            int type = metadataBean.getType();
            if (type == 1) {
                sb.append(metadataBean.getFieldFixedValue());
            } else {
                if (metadataBean.getFieldGroupIndex() == FieldOfObj.BILL_INFO) {
                    sb.append(processMainRemark(billInfo, rule, metadataBean));
                } else {
                    if (rule.isRemarkDuplicateFlag()) {
                        sb.append(processItemRemark(billItems, rule, metadataBean));
                    }
                }
            }

            if (i != remarkFieldMetadataBeanList.size() - 1) {
                concatSeparator(rule, sb);
            }

        }

        if (!rule.isRemarkDuplicateFlag()) {

            if (Objects.nonNull(sb) && sb.length() > 0) {
                if (StringUtils.endsWith(sb.toString(), SPACE)) {
                    sb.replace(sb.lastIndexOf(SPACE), sb.length(), TAB);
                } else {
                    sb.append(TAB);
                }
            }

            sb.append(processDuplicateItemRemark(billItems, rule, remarkFieldMetadataBeanList));
        }

        return sb.toString();
    }

    /**
     * 组装原生的备注
     * @param billInfo
     * @param remarkFieldMetadataBeanList
     * @return
     */
    protected String handleOriginalRemark(BillInfo billInfo, SplitRule rule, List<RemarkFieldMetadata> remarkFieldMetadataBeanList) {

        StringBuilder sb = new StringBuilder();

        //业务单备注
        long size = checkMainRemark(remarkFieldMetadataBeanList);

        if (size > 0) {
            appendSpace(sb);
            sb.append(Objects.toString(billInfo.getRemark(), ""));
            boolean remarkWrapsFlag = rule.getRemarkWrapsFlag();
            if (remarkWrapsFlag) {
                appendTab(sb);
            }
        }

        appendSpace(sb);

        return sb.toString();
    }

    /**
     * 组装定制的备注
     * @param billInfo
     * @param rule
     * @param remarkFieldMetadataBeanList
     * @return
     */
    protected String handleCustomRemark(BillInfo billInfo, SplitRule rule, List<RemarkFieldMetadata> remarkFieldMetadataBeanList) {

        List<BillItem> billItems = billInfo.getBillItems();

        /**
         * 解读remarkDuplicateFlag
         *
         * true:
         * 按配置的顺序拼接主信息字段或明细信息字段，并且明细备注内容去重
         * ex: mainKey1:value1  mainKey2:value2  itemKey1: distinctValues1  itemKey2: distinctValues2 ..
         *
         * false:
         * 按主信息备注字段先拼接备注，换行后，拼接 每个明细对应key-value
         * ex: mainKey1:value1 mainKey2:value2 \n
         *    itemKey1: items[0].value1  itemKey2: items[0].value2 \n
         *    itemKey1: items[1].value1  itemKey2: items[1].value2 \n
         *    ..
         */

        StringBuilder sb = new StringBuilder();

        for (RemarkFieldMetadata metadataBean : remarkFieldMetadataBeanList) {
            if (metadataBean.getFieldGroupIndex() == FieldOfObj.BILL_INFO) {

                /**
                 * 跳过原生备注处理过程
                 */
                if (StringUtils.equals(metadataBean.getFieldName(), RemarkFieldNameEnum.REMARK.getValue())) {
                    continue;
                }

                sb.append(processMainRemark(billInfo, rule, metadataBean));

            } else {
                //明细备注
                if (rule.isRemarkDuplicateFlag()) {
                    sb.append(processItemRemark(billItems, rule, metadataBean));
                }
            }

            //最后一个规则备注不加","
            boolean remarkWrapsFlag = rule.getRemarkWrapsFlag();
            if (remarkWrapsFlag) {
                appendTab(sb);
            } else {
                appendSpace(sb);
            }

        }

        if (!rule.isRemarkDuplicateFlag()) {
            if (Objects.nonNull(sb) && sb.length() > 0) {
                if (StringUtils.endsWith(sb.toString(), SPACE)) {
                    sb.replace(sb.lastIndexOf(SPACE), sb.length(), TAB);
                } else {
                    sb.append(TAB);
                }
            }

            sb.append(processDuplicateItemRemark(billItems, rule, remarkFieldMetadataBeanList));
        }

        return sb.toString();
    }

    protected void concatSeparator(SplitRule rule, StringBuilder sb) {

        Integer separatorType = rule.getRemarkSeparatorType();
        if (separatorType == null) {

            //最后一个规则备注不加","
            boolean remarkWrapsFlag = rule.getRemarkWrapsFlag();
            if (remarkWrapsFlag) {
                appendTab(sb);
            } else {
                appendSpace(sb);
            }
        } else {
            switch (separatorType) {
                case 1:
                    appendComma(sb);
                    break;
                case 2:
                    appendDawn(sb);
                    break;
                case 3:
                    appendColon(sb);
                    break;
                case 4:
                    appendSemiColon(sb);
                    break;
                case 5:
                    appendDash(sb);
                    break;
                case 6:
                    appendVerticalBar(sb);
                    break;
                case 7:
                    appendSpace(sb);
                    break;
                case 8:
                    appendTab(sb);
                    break;
            }

        }
    }

    protected String getSeparator(SplitRule rule) {
        String separator = SPACE;
        Integer separatorType = rule.getRemarkSeparatorType();
        if(null == separatorType) {
            return separator;
        }
        switch (separatorType) {
            case 1:
                separator = COMMA;
                break;
            case 2:
                separator = DAWN;
                break;
            case 3:
                separator = COLON;
                break;
            case 4:
                separator = SEMICOLON;
                break;
            case 5:
                separator = DASH;
                break;
            case 6:
                separator = VERTICAL_BAR;
                break;
            case 7:
                separator = SPACE;
                break;
            case 8:
                separator = TAB;
                break;
        }
        return separator;
    }


    protected long checkMainRemark(List<RemarkFieldMetadata> remarkFieldMetadataBeanList) {

        return remarkFieldMetadataBeanList.stream()
                .filter(remarkFieldMetadata ->
                        RemarkFieldNameEnum.REMARK.getValue().equalsIgnoreCase(remarkFieldMetadata.getFieldName()) &&
                                FieldOfObj.BILL_INFO == remarkFieldMetadata.getFieldGroupIndex())
                .count();
    }

    /**
     * 处理主信息字段并拼接备注
     * @param billItems
     * @param metadataBean
     * @return
     */
    protected String processItemRemark(List<BillItem> billItems, SplitRule splitRule, RemarkFieldMetadata metadataBean) {

        StringBuilder sb = new StringBuilder();
        String value;
        boolean removeRemarkItemDuplicate = splitRule.isRemoveRemarkItemDuplicate();

        if (StringUtils.equals(metadataBean.getFieldName(), RemarkFieldNameEnum.TAX_PRE.getValue())) {

            value = billItems.stream().
                    map(billItem -> this.getTaxPre(billItem.getTaxPre())).filter(StringUtils::isNotBlank).distinct().collect(Collectors.joining(this.getSeparator(splitRule)));

        } else if (StringUtils.equals(metadataBean.getFieldName(), RemarkFieldNameEnum.ZERO_TAX.getValue())) {

            value = billItems.stream().
                    map(billItem -> this.getZeroTaxMark(billItem.getZeroTax())).filter(StringUtils::isNotBlank).distinct().collect(Collectors.joining(this.getSeparator(splitRule)));

        } else {

            if (removeRemarkItemDuplicate) {
                value = removeItemDuplicate(billItems, splitRule, metadataBean);

            } else {
                value = billItems.stream()
                        .map(billItem -> this.getCustomFieldValue(billItem, metadataBean.getFieldName())).filter(StringUtils::isNotBlank).distinct()
                        .collect(Collectors.joining(this.getSeparator(splitRule)));

            }

        }

        String key = getDefault(metadataBean.getFieldDisplayName());
        sb.append(assembleRemarkUnit(key, value, splitRule));

        return sb.toString();
    }

    /**
     * 明细内容去重再合并
     * item1.ext1: a,b
     * item2.ext1: a,c
     * -> ext1: a,b,c
     * <p>
     * item1.ext1: a,b,c
     * item2.ext2: b,c,d
     * -> ext1: a,b,c,d
     */
    protected String removeItemDuplicate(List<BillItem> billItems,SplitRule splitRule, RemarkFieldMetadata metadataBean) {

        String value = billItems.stream()
                .map(billItem -> {

                    List<String> results = new ArrayList<>();

                    String eachValue = this.getCustomFieldValue(billItem, metadataBean.getFieldName());

                    if (eachValue.contains(",")) {
                        String[] strings = eachValue.split(",");
                        results.addAll(Arrays.asList(strings));
                    } else {
                        results.add(eachValue);
                    }
                    return results;

                }).flatMap(Collection::stream).filter(StringUtils::isNotBlank).distinct()
                .collect(Collectors.joining(this.getSeparator(splitRule)));

        return value;

    }

    /**
     * 处理不去重的明细备注
     * key1:value1 key2:value2 备注拼接方式
     * @param billItems
     * @param metadataBeans
     */
    protected String processDuplicateItemRemark(List<BillItem> billItems, SplitRule splitRule, List<RemarkFieldMetadata> metadataBeans) {

        StringBuilder sb = new StringBuilder();

        billItems.forEach(billItem -> {

            for (RemarkFieldMetadata metadataBean : metadataBeans) {

                if (metadataBean.getFieldGroupIndex() == FieldOfObj.BILL_INFO) {
                    continue;
                }

                String value;
                if (StringUtils.equals(metadataBean.getFieldName(), RemarkFieldNameEnum.TAX_PRE.getValue())) {
                    value = this.getTaxPre(billItem.getTaxPre());

                } else if (StringUtils.equals(metadataBean.getFieldName(), RemarkFieldNameEnum.ZERO_TAX.getValue())) {
                    value = this.getZeroTaxMark(billItem.getZeroTax());

                } else if (RemarkFieldNameEnum.isAmountKindField(metadataBean.getFieldName())) {
                    value = getCustomFieldValue(billItem, metadataBean.getFieldName());
                    BigDecimal amount = Optional.ofNullable(CommonTools.getBigDecimal(value))
                            .map(e -> e.setScale(2, RoundingMode.HALF_UP)).orElse(null);

                    value = amount != null ? amount.toString() : "";

                } else {
                    value = this.getCustomFieldValue(billItem, metadataBean.getFieldName());
                }

                String key = getDefault(metadataBean.getFieldDisplayName());
                sb.append(assembleRemarkUnit(key, value, splitRule));

            }

            sb.append(TAB);
        });

        return sb.toString();
    }

    /**
     * 处理主信息字段并拼接备注
     * @param billInfo
     * @param rule
     * @param metadataBean
     */
    protected String processMainRemark(BillInfo billInfo, SplitRule rule, RemarkFieldMetadata metadataBean) {

        StringBuilder sb = new StringBuilder();

        if (StringUtils.equals(metadataBean.getFieldName(), RemarkFieldNameEnum.FIXED_REMARK.getValue())) {

            sb.append(Strings.nullToEmpty(rule.getFixedRemarkText()));

        } else if (StringUtils.equals(metadataBean.getFieldName(), RemarkFieldNameEnum.REMARK.getValue())) {

            sb.append(Strings.nullToEmpty(billInfo.getRemark()));

        } else if (StringUtils.equals(metadataBean.getFieldName(), RemarkFieldNameEnum.PRICE_METHOD.getValue())) {

            PriceMethod priceMethod = rule.getPriceMethod();

            String key = getDefault(metadataBean.getFieldDisplayName());
            String value = this.getPriceMethod(priceMethod);

            sb.append(assembleRemarkUnit(key, value, rule));

        } else if (RemarkFieldNameEnum.isAmountKindField(metadataBean.getFieldName())) {

            String key = getDefault(metadataBean.getFieldDisplayName());
            String value = getCustomFieldValue(billInfo, metadataBean.getFieldName());

            BigDecimal amount = Optional.ofNullable(CommonTools.getBigDecimal(value)).map(e -> e.setScale(2, RoundingMode.HALF_UP)).orElse(null);
            String formatValue = amount != null ? amount.toString() : "";

            sb.append(assembleRemarkUnit(key, formatValue, rule));

        } else {

            String key = getDefault(metadataBean.getFieldDisplayName());
            String value = getCustomFieldValue(billInfo, metadataBean.getFieldName());

            sb.append(assembleRemarkUnit(key, value, rule));

        }

        return sb.toString();
    }

    private String getDefault(String key) {

        return StringUtils.isEmpty(key) ? StringUtils.EMPTY : key;
    }

    protected String assembleRemarkUnit(String key, String value, SplitRule rule) {

        boolean hideRemarkFieldName = rule.isHideRemarkFieldName();

        StringBuilder sb = new StringBuilder();
        if(rule.getRemarkKeyValueShowType()==null){
            rule.setRemarkKeyValueShowType(0);
        }
        if (StringUtils.isNotBlank(value)) {
            switch (rule.getRemarkKeyValueShowType()){
                case 1:
                    if (hideRemarkFieldName) {
                        sb.append(COMMA).append(value);
                    } else {
                        sb.append(key).append(COMMA).append(value);
                    }
                    break;
                case 2:
                    if (hideRemarkFieldName) {
                        sb.append(DAWN).append(value);
                    } else {
                        sb.append(key).append(DAWN).append(value);
                    }
                    break;
                case 3:
                    if (hideRemarkFieldName) {
                        sb.append(COLON).append(value);
                    } else {
                        sb.append(key).append(COLON).append(value);
                    }
                    break;
                case 4:
                    if (hideRemarkFieldName) {
                        sb.append(SEMICOLON).append(value);
                    } else {
                        sb.append(key).append(SEMICOLON).append(value);
                    }
                    break;
                case 5:
                    if (hideRemarkFieldName) {
                        sb.append(DASH).append(value);
                    } else {
                        sb.append(key).append(DASH).append(value);
                    }
                    break;
                case 6:
                    if (hideRemarkFieldName) {
                        sb.append(VERTICAL_BAR).append(value);
                    } else {
                        sb.append(key).append(VERTICAL_BAR).append(value);
                    }
                    break;
                case 7:
                    if (hideRemarkFieldName) {
                        sb.append(SPACE).append(value);
                    } else {
                        sb.append(key).append(SPACE).append(value);
                    }
                    break;
                case 8:
                    if (hideRemarkFieldName) {
                        sb.append(TAB).append(value);
                    } else {
                        sb.append(key).append(TAB).append(value);
                    }
                    break;
                case 9:
                    if (hideRemarkFieldName) {
                        sb.append(SLASH).append(value);
                    } else {
                        sb.append(key).append(SLASH).append(value);
                    }
                    break;
                default:
                    if (hideRemarkFieldName) {
                        sb.append("[").append(value).append("]");
                    } else {
                        sb.append(key).append("[").append(value).append("]");
                    }
                    break;
            }
        }

        return sb.toString();
    }

    protected String handleRedInvoiceRemark(BillInfo billInfo) {

        StringBuilder sb = new StringBuilder();

        if (InvoiceType.SPECIAL.value().equals(billInfo.getInvoiceType())) {
            if (StringUtils.isNotBlank(billInfo.getRedNotificationNo())) {
                sb.append(CommonTools.format(RED_SPECIAL_REMARK, billInfo.getRedNotificationNo()));
            }

        } else if (InvoiceType.NORMAL.value().equals(billInfo.getInvoiceType()) ||
                InvoiceType.ELECTRONIC.value().equals(billInfo.getInvoiceType())) {

            if (StringUtils.isNotBlank(billInfo.getOriginInvoiceCode())) {
                sb.append(RED_NORMAL_REMARK).append(CommonTools.format(RED_NORMAL_REMARK_CODE, billInfo.getOriginInvoiceCode()));
                if (StringUtils.isNotBlank(billInfo.getOriginInvoiceNo())) {
                    sb.append(CommonTools.format(RED_NORMAL_REMARK_NO, billInfo.getOriginInvoiceNo()));
                }

            } else if (StringUtils.isNotBlank(billInfo.getOriginInvoiceNo())) {
                sb.append(RED_NORMAL_REMARK).append(CommonTools.format(RED_NORMAL_REMARK_NO, billInfo.getOriginInvoiceNo()));
            }

        }

        return sb.toString();
    }

    private void appendSpace(StringBuilder sb) {
        if (StringUtils.isNotBlank(sb.toString()) && !StringUtils.endsWith(sb.toString(), SPACE)) {
            sb.append(SPACE);
        }
    }

    private void appendTab(StringBuilder sb) {
        if (StringUtils.isNotBlank(sb.toString()) && !StringUtils.endsWith(sb.toString(), TAB)) {
            sb.append(TAB);
        }
    }

    private void appendComma(StringBuilder sb) {
        if (StringUtils.isNotBlank(sb.toString()) && !StringUtils.endsWith(sb.toString(), COMMA)) {
            sb.append(COMMA);
        }
    }

    private void appendDawn(StringBuilder sb) {
        if (StringUtils.isNotBlank(sb.toString()) && !StringUtils.endsWith(sb.toString(), DAWN)) {
            sb.append(DAWN);
        }
    }

    private void appendColon(StringBuilder sb) {
        if (StringUtils.isNotBlank(sb.toString()) && !StringUtils.endsWith(sb.toString(), COLON)) {
            sb.append(COLON);
        }
    }

    private void appendSemiColon(StringBuilder sb) {
        if (StringUtils.isNotBlank(sb.toString()) && !StringUtils.endsWith(sb.toString(), SEMICOLON)) {
            sb.append(SEMICOLON);
        }
    }

    private void appendDash(StringBuilder sb) {
        if (StringUtils.isNotBlank(sb.toString()) && !StringUtils.endsWith(sb.toString(), DASH)) {
            sb.append(DASH);
        }
    }

    private void appendVerticalBar(StringBuilder sb) {
        if (StringUtils.isNotBlank(sb.toString()) && !StringUtils.endsWith(sb.toString(), VERTICAL_BAR)) {
            sb.append(VERTICAL_BAR);
        }
    }


    /**
     * 获取发票主信息或明细信息字段值
     * @param object
     * @param fieldName
     * @return
     */
    protected String getCustomFieldValue(Object object, String fieldName) {

        String text = "";
        if (object instanceof BillItem) {
            text = "明细信息";
        } else if (object instanceof BillInfo) {
            text = "主信息";
        }

        try {
            return Objects.toString(FieldUtils.getFieldValue(object, fieldName), "");
        } catch (ReflectionException e) {
            logger.warn("元数据获取{}字段值错误:{}", text, e.getMessage());
        }

        return "";
    }

    /**
     * 是否享受税收优惠政策转换
     * @param taxPre 是否享受税收优惠政策 0 -不 1-享受
     * @return 是否享受税收优惠政策 中文
     */
    private String getTaxPre(String taxPre) {
        return StringUtils.equals(taxPre, "1") ? "享受" : "不享受";
    }

    /**
     * 零税率标识转换
     * @param zeroTax 零税率标识
     * @return 零税率标识中文
     */
    private String getZeroTaxMark(String zeroTax) {
        return StringUtils.equals(zeroTax, "0") ? "出口退税"
                : StringUtils.equals(zeroTax, "1") ? "免税"
                : StringUtils.equals(zeroTax, "2") ? "不征税"
                : StringUtils.equals(zeroTax, "3") ? "普通零税率"
                : "非零税率";
    }

    /**
     * 价格方式转换
     * @param priceMethod 价格方式 1-含税 2-不含税
     * @return 价格方式中文
     */
    private String getPriceMethod(PriceMethod priceMethod) {
        return priceMethod == PriceMethod.WITH_TAX ? "含税" : "不含税";

    }
}
