package com.xforceplus.phoenix.split.checker;

import com.xforceplus.phoenix.split.domain.ItemAmountInfo;
import com.xforceplus.phoenix.split.model.BillItem;
import com.xforceplus.phoenix.split.model.SplitRule;
import com.xforceplus.phoenix.split.util.BigDecimalUtil;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;

@Component
public class SplitLimitChecker {

    /**
     * 超出任意限额
     */
    public boolean gtLimit(BillItem billItem, SplitRule rule) {
        return this.gtLimitAmount(billItem, rule) || this.gtLimitQuantity(billItem, rule);
    }

    /**
     * 超出金额限额
     */
    private boolean gtLimitAmount(BillItem billItem, SplitRule rule) {
        BigDecimal amount;
        if (rule.isLimitIsAmountWithTax()) {
            //含税金额
            amount = billItem.getAmountWithTax().subtract(billItem.getInnerPrepayAmountWithTax())
                    .subtract(billItem.getInnerDiscountWithTax())
                    .subtract(billItem.getOutterDiscountWithTax())
                    .subtract(billItem.getOutterPrepayAmountWithTax());
        } else {
            //不含税金额
            amount = billItem.getAmountWithoutTax().subtract(billItem.getInnerDiscountWithoutTax())
                    .subtract(billItem.getInnerPrepayAmountWithoutTax())
                    .subtract(billItem.getOutterDiscountWithoutTax())
                    .subtract(billItem.getOutterPrepayAmountWithoutTax());
        }
        return amount.compareTo(rule.getInvoiceLimit()) > 0;
    }

    /**
     * 超出数量限额
     */
    private boolean gtLimitQuantity(BillItem billItem, SplitRule rule) {
        if (MapUtils.isEmpty(rule.getQuantityLimitMap()) || !rule.getQuantityLimitMap().containsKey(billItem.getQuantityUnit())
                || StringUtils.isBlank(billItem.getQuantityUnit())) {
            return false;
        }
        if (BigDecimalUtil.nullToZero(billItem.getQuantity()).compareTo(BigDecimal.ZERO) == 0) {
            return false;
        }
        return billItem.getQuantity().compareTo(rule.getQuantityLimitMap().get(billItem.getQuantityUnit())) > 0;
    }

    /**
     * 超出任意限额
     */
    public boolean gtLimit(ItemAmountInfo itemAmountInfo, SplitRule rule, BigDecimal quantityLimit) {
        return this.gtLimitAmount(itemAmountInfo, rule) || this.gtLimitQuantity(itemAmountInfo, quantityLimit);
    }

    /**
     * 超出金额限额
     */
    private boolean gtLimitAmount(ItemAmountInfo itemAmountInfo, SplitRule rule) {
        BigDecimal amount;
        if (rule.isLimitIsAmountWithTax()) {
            //含税金额
            amount = itemAmountInfo.getAmountWithTax().subtract(itemAmountInfo.getDiscountWithTax());
        } else {
            //不含税金额
            amount = itemAmountInfo.getAmountWithoutTax().subtract(itemAmountInfo.getDiscountWithoutTax());
        }
        return amount.compareTo(rule.getInvoiceLimit()) > 0;
    }

    /**
     * 超出数量限额
     */
    private boolean gtLimitQuantity(ItemAmountInfo itemAmountInfo, BigDecimal quantityLimit) {
        return itemAmountInfo.getQuantity().compareTo(quantityLimit) > 0;
    }
}
