package com.xforceplus.phoenix.split.service;

import com.google.common.base.Stopwatch;
import com.google.common.collect.Lists;
import com.xforceplus.phoenix.split.constant.*;
import com.xforceplus.phoenix.split.exception.SplitBizException;
import com.xforceplus.phoenix.split.model.BillInfo;
import com.xforceplus.phoenix.split.model.BillItem;
import com.xforceplus.phoenix.split.model.CreatePreInvoiceParam;
import com.xforceplus.phoenix.split.model.ItemTypeCodeEnum;
import com.xforceplus.phoenix.split.model.PriceMethod;
import com.xforceplus.phoenix.split.model.RedReasonEnum;
import com.xforceplus.phoenix.split.model.SplitPreInvoiceInfo;
import com.xforceplus.phoenix.split.model.SplitRule;
import com.xforceplus.phoenix.split.service.impl.Bill2PreInvoiceService;
import java.util.Optional;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.io.UnsupportedEncodingException;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import static com.xforceplus.phoenix.split.service.dataflow.impl.InvoiceLimitProcessPlugin.ERROR_AMOUNT;
import static com.xforceplus.phoenix.split.service.impl.DefaultSplitBillItemAmountServiceImpl.MIN_QUANTITY;

/**
 * @Author chenlingwei
 * @create 2023/3/27 5:42 PM
 */

@Service
public class SplitMainHandleService {

    private static final int DEFAULT_UNIT_PRICE_SCALE = 15;
    public static final String UNIT_PRICE_ERROR_MESSAGE = "发票单价必填";
    public static final String QUANTITY_UNIT_ERROR_MESSAGE = "发票单位必填";
    public static final String QUANTITY_UNIT_ERROR_VALUE_MESSAGE = "发票单位有误";
    public static final String QUANTITY_ERROR_MESSAGE = "发票数量必填";
    public static final String BUILDING_ADDRESS_KEY = "placeOfOccurrence";
    public static final String PROPERTY_ADDRESS_KEY = "realEstateAddress";
    public static final String JDC_INVOICE_TYPE_KEY = "invoiceType";
    public static final String JDC_ITEM_SPEC_KEY = "itemSpec";
    public static final String JDC_ITEM_QUANTITY_UNIT = "辆";
    public static final String JDC_INVOICE_TYPE_ERROR_MESSAGE = "当前开具的发票为机动车专用发票，请修改发票类型后提交";
    public static final String JDC_ITEM_SPEC_ERROR_MESSAGE = "当前开具的发票为机动车专用发票，必须填写规格型号";
    public static final String JDC_ITEM_QUANTITY_ERROR_MESSAGE = "当前开具的发票为机动车专用发票，数量必须为非零整数";
    public static final String JDC_ITEM_QUANTITY_UNIT_ERROR_MESSAGE = "当前开具的发票为机动车专用发票，数量单位必须为辆";

    public static final String GOODS_TAX_NO="3040502020200000000";

    private Logger logger = LoggerFactory.getLogger(getClass());

    @Autowired
    private Bill2PreInvoiceService bill2PreInvoiceService;
    @Autowired
    private BasicDataProcessService basicDataProcessService;
    /**
     * 临时方案 1- 税控&&老税编 || 数电&&新税编
     */
    @Value("${config.goods-tax-no.validation.mode}")
    private String goodsTaxNoMode;

    public List<SplitPreInvoiceInfo> splitAndAssembleInvoice(CreatePreInvoiceParam createPreInvoiceParam, TaxDeviceType taxDeviceType, String txId) {

        Stopwatch stopwatch = Stopwatch.createStarted();

        processBillItem(createPreInvoiceParam.getBillInfo(), createPreInvoiceParam.getRule(), txId);

        List<SplitPreInvoiceInfo> data = bill2PreInvoiceService.createPreInvoiceFromBill(
                createPreInvoiceParam.getBillInfo(),
                createPreInvoiceParam.getRule(),
                StringUtils.isBlank(createPreInvoiceParam.getRuleCode()) ? "default" : createPreInvoiceParam.getRuleCode(),
                taxDeviceType);

        logger.info("SplitMainHandleService.splitAndAssembleInvoice.generate pre_invoice txId = {} elapsed time = {} ms, pre invoice size:{}", txId, stopwatch.elapsed().toMillis(), data.size());

        return data;
    }

    public void processBillItem(BillInfo billInfo, SplitRule rule, String txId) {

        Stopwatch stopwatch = Stopwatch.createStarted();

        if (rule.getUnitPriceScale() == null) {
            rule.setUnitPriceScale(DEFAULT_UNIT_PRICE_SCALE);
        }

        List<BillItem> items = billInfo.getBillItems();
        PriceMethod priceMethod = PriceMethod.value(billInfo.getPriceMethod());
        rule.setPriceMethod(priceMethod);

        /**
         * 明细处理全逻辑
         * https://xforceplus.yuque.com/dew9bm/vk8akw/vqrtq1nnwlz5p75d
         */
        for (BillItem billItem : items) {

            adjustBillItem(billItem, rule);

            validateInvoiceItem(billItem, rule);

            validateSpecialInvoice(billInfo, billItem, rule);

            handleItemAmount(billItem, rule, priceMethod);
        }

        reCalculateTotalAmount(rule, billInfo, items);

        logger.info("SplitMainHandleService.processBillItem txId={} elapsed time = {} ms, billItem size:{}", txId, stopwatch.elapsed().toMillis(), items.size());
    }

    /**
     * 金额类处理
     *
     * @param billItem
     * @param rule
     * @param priceMethod
     */
    public void handleItemAmount(BillItem billItem, SplitRule rule, PriceMethod priceMethod) {

        //隐藏价外折扣配置处理
        calculateInnerAmount(billItem, rule);

        //金额公式校验
        validateItemAmount(billItem, rule);

        //重新金额计算
        reCalculateOriginalTaxAmount(rule, billItem, priceMethod);

        //空实现
        reCalculateUnitPrice(billItem, rule);

        basicDataProcessService.adjustUnitPrice(billItem, rule);

    }

    private void adjustBillItem(BillItem billItem, SplitRule rule) {

        //格式化单价
        adjustBillItemAmount(billItem);

        //商品名称处理
        adjustBillItemName(billItem, rule);

        //税编处理（新老税编）
        adjustBillItemGoodsTaxNo(billItem, rule);

    }

    public void adjustBillItemGoodsTaxNo(BillItem billItem, SplitRule rule) {

        String taxInvoiceSource = rule.getTaxInvoiceSource();

        if (SpecialGoodsTaxNoEnum.isIntangibleAssets(billItem.getGoodsTaxNo())) {

            if (TaxInvoiceSourceEnum.QD.getValue().equals(taxInvoiceSource)) {
                if (StringUtils.isNotBlank(goodsTaxNoMode) && goodsTaxNoMode.equals("1")) {
                    billItem.setGoodsTaxNo(SpecialGoodsTaxNoEnum.INTANGIBLE_ASSETS_OLD.getValue());
                } else {
                    billItem.setGoodsTaxNo(SpecialGoodsTaxNoEnum.INTANGIBLE_ASSETS_NEW.getValue());
                }

            } else if (TaxInvoiceSourceEnum.SK.getValue().equals(taxInvoiceSource)) {
                billItem.setGoodsTaxNo(SpecialGoodsTaxNoEnum.INTANGIBLE_ASSETS_NEW.getValue());
            }
        }

    }

    /**
     * 去除商品名称中的特殊字符
     *
     * @param billItem
     * @param rule
     */
    public void adjustBillItemName(BillItem billItem, SplitRule rule) {

        boolean removeIllegalItemNameCharacter = rule.isRemoveIllegalItemNameCharacter();
        if (removeIllegalItemNameCharacter) {
            String itemName = billItem.getItemName();

            try {
                String gbkItemName = new String(itemName.getBytes("gbk"), "gbk");
                String utf8ItemName = new String(itemName.getBytes(StandardCharsets.UTF_8), StandardCharsets.UTF_8);

                if (!gbkItemName.equals(utf8ItemName)) {

                    StringBuilder sb = new StringBuilder();
                    char[] gbkChars = gbkItemName.toCharArray();
                    char[] utf8Chars = utf8ItemName.toCharArray();
                    for (int i = 0; i < gbkChars.length; i++) {
                        char gbkChar = gbkChars[i];
                        char utfChar = utf8Chars[i];
                        if (gbkChar == utfChar) {
                            sb.append(gbkChar);
                        }
                    }
                    billItem.setItemName(sb.toString().trim());
                }

            } catch (UnsupportedEncodingException e) {
                logger.error(e.getMessage(), e);
            }
        }

    }

    /**
     * 明细金额调整
     *
     * @param billItem
     */
    public void adjustBillItemAmount(BillItem billItem) {

        BigDecimal unitPrice = billItem.getUnitPrice();
        billItem.setUnitPrice(unitPrice.stripTrailingZeros());

    }

    /**
     * 数电&&含税 自动反算不含税单价
     *
     * @param billItem
     * @param rule
     */
    public void reCalculateUnitPrice(BillItem billItem, SplitRule rule) {
    }

    /**
     * 校验特殊发票
     *
     * @param billInfo
     */
    public void validateSpecialInvoice(BillInfo billInfo, BillItem billItem, SplitRule rule) {

        String redReason = billInfo.getRedReason();

        RedReasonEnum redReasonEnum = !org.springframework.util.StringUtils.isEmpty(redReason) ? RedReasonEnum.fromValue(redReason) : null;

        this.validateItemTypeCodeCPY(billItem, redReasonEnum);

        this.validateItemTypeCodeKCP(billItem, redReasonEnum);

        this.validateSpecialItemTypeCode(billItem, rule);

    }

    /**
     * 成品油校验逻辑
     * 开成品油蓝字发票 数量单价都不能为空或者0
     * 开成品油红字发票 红冲原因不为空&&不为销售折让 数量单价都不能为空或者0
     *
     * @param billItem
     * @param redReasonEnum
     */
    public void validateItemTypeCodeCPY(BillItem billItem, RedReasonEnum redReasonEnum) {

        String itemTypeCode = billItem.getItemTypeCode();

        boolean isRedItem = billItem.getAmountWithoutTax().compareTo(BigDecimal.ZERO) < 0;

        if (ItemTypeCodeEnum.OIL.getValue().equals(itemTypeCode) &&
                (!isRedItem || (redReasonEnum != null && RedReasonEnum.SALES_ALLOWANCE != redReasonEnum))) {

            if (billItem.getQuantity() == null || BigDecimal.ZERO.compareTo(billItem.getQuantity()) == 0) {
                throw new SplitBizException(ItemTypeCodeEnum.OIL.getDescription() + QUANTITY_ERROR_MESSAGE);
            }

            if (billItem.getUnitPrice() == null || BigDecimal.ZERO.compareTo(billItem.getUnitPrice()) == 0) {
                throw new SplitBizException(ItemTypeCodeEnum.OIL.getDescription() + UNIT_PRICE_ERROR_MESSAGE);
            }

            if (org.springframework.util.StringUtils.isEmpty(billItem.getQuantityUnit())) {
                throw new SplitBizException(ItemTypeCodeEnum.OIL.getDescription() + QUANTITY_UNIT_ERROR_MESSAGE);
            }

        }
    }

    /**
     * https://xforceplus.yuque.com/dew9bm/tuk5l5/noiofik6eaflsuzh
     * 数电机动车票校验规则
     */
    public void validateItemTypeCodeJDC(BillItem billItem, String invoiceType, SplitRule rule) {
        String itemTypeCode = billItem.getItemTypeCode();
        String taxInvoiceSource = rule.getTaxInvoiceSource();
        if (StringUtils.equals(TaxInvoiceSourceEnum.QD.getValue(), taxInvoiceSource)
                && StringUtils.equals(ItemTypeCodeEnum.VEHICLE.getValue(), itemTypeCode) && Lists.newArrayList(InvoiceType.SE.value(), InvoiceType.SPECIAL.value()).contains(invoiceType)) {
            if (StringUtils.isBlank(billItem.getItemSpec())) {
                throw new SplitBizException(JDC_ITEM_SPEC_ERROR_MESSAGE);
            }
            if (!BigDecimalUtils.isQuanlityOfJdc(billItem.getQuantity())) {
                throw new SplitBizException(JDC_ITEM_QUANTITY_ERROR_MESSAGE);
            }
            if (!StringUtils.equals(JDC_ITEM_QUANTITY_UNIT, billItem.getQuantityUnit())) {
                throw new SplitBizException(JDC_ITEM_QUANTITY_UNIT_ERROR_MESSAGE);
            }
        }
    }


    /**
     * 矿产品校验逻辑
     * 针对【矿产品】的蓝字拆票，需校验其单位、 数量、单价不能为空，且单位仅可为：吨、立方米、千克、克或克拉
     * 针对【矿产品】的红字拆票，当红冲原因=【销货退回、开票有误、服务中止】时，单位与数量必填
     * 当红冲原因=【销售折让】时，单位、数量可为空
     *
     * @param billItem
     * @param redReasonEnum
     */
    public void validateItemTypeCodeKCP(BillItem billItem, RedReasonEnum redReasonEnum) {

        String itemTypeCode = billItem.getItemTypeCode();
        String goodsTaxNo = billItem.getGoodsTaxNo();

        if (ItemTypeCodeEnum.MINERALS.getValue().equals(itemTypeCode)) {

            boolean isRedItem = billItem.getAmountWithoutTax().compareTo(BigDecimal.ZERO) < 0;

            if (!isRedItem) {

                if (billItem.getQuantity() == null || BigDecimal.ZERO.compareTo(billItem.getQuantity()) == 0) {
                    throw new SplitBizException(ItemTypeCodeEnum.MINERALS.getDescription() + QUANTITY_ERROR_MESSAGE);
                }

                if (billItem.getUnitPrice() == null || BigDecimal.ZERO.compareTo(billItem.getUnitPrice()) == 0) {
                    throw new SplitBizException(ItemTypeCodeEnum.MINERALS.getDescription() + UNIT_PRICE_ERROR_MESSAGE);
                }

                if (org.springframework.util.StringUtils.isEmpty(billItem.getQuantityUnit())) {
                    throw new SplitBizException(ItemTypeCodeEnum.MINERALS.getDescription() + QUANTITY_UNIT_ERROR_MESSAGE);
                }

                /**
                 * 特殊税编才能使用【桶】为单位
                 */
                if (!ProductMinerals.getMineralQuantityUnits().contains(billItem.getQuantityUnit()) ||
                        ("桶".equals(billItem.getQuantityUnit()) && !ProductMinerals.getMineralOilGoodsTaxNos().contains(goodsTaxNo))) {
                    throw new SplitBizException(ItemTypeCodeEnum.MINERALS.getDescription() + QUANTITY_UNIT_ERROR_VALUE_MESSAGE);
                }

            } else {

                if (redReasonEnum != null && RedReasonEnum.SALES_ALLOWANCE != redReasonEnum) {

                    if (billItem.getQuantity() == null || BigDecimal.ZERO.compareTo(billItem.getQuantity()) == 0) {
                        throw new SplitBizException(ItemTypeCodeEnum.MINERALS.getDescription() + QUANTITY_ERROR_MESSAGE);
                    }

                    if (org.springframework.util.StringUtils.isEmpty(billItem.getQuantityUnit())) {
                        throw new SplitBizException(ItemTypeCodeEnum.MINERALS.getDescription() + QUANTITY_UNIT_ERROR_MESSAGE);
                    }
                }
            }

        }
    }

    /**
     * 数电校验特殊服务发票明细
     *
     * @param billItem
     * @param rule
     */
    public void validateSpecialItemTypeCode(BillItem billItem, SplitRule rule) {

        List<Map<String, Object>> specialAdditions = billItem.getSpecialAdditions();
        boolean isRedItem = billItem.getAmountWithoutTax().compareTo(BigDecimal.ZERO) < 0;
        if (rule.getTaxInvoiceSource().equals(TaxInvoiceSourceEnum.QD.getValue()) && ItemTypeCodeEnum.isAllElectronicPropertySpecialInvoice(billItem.getItemTypeCode())) {
            List<Map<String, Object>> additions = billItem.getSpecialAdditions();
            Optional.ofNullable(additions)
                .filter(list -> !list.isEmpty())
                .map(list -> list.get(0).get("areaUnit"))
                .map(areaUnitObj -> SpecialAreaUnitEnum.getEnum(String.valueOf(areaUnitObj)))
                .ifPresent(areaUnitEnum -> billItem.setQuantityUnit(areaUnitEnum.getDescription()));
        }
        if (rule.getTaxInvoiceSource().equals(TaxInvoiceSourceEnum.QD.getValue()) &&
                ItemTypeCodeEnum.isAllElectronicSpecialInvoice(billItem.getItemTypeCode()) &&
                !isRedItem) {

            if (CollectionUtils.isEmpty(specialAdditions)) {
                throw new SplitBizException("特殊业务字段不能为空 ");
            }

            if (ItemTypeCodeEnum.PROPERTY_RENT.getValue().equals(billItem.getItemTypeCode()) ||
                    ItemTypeCodeEnum.PROPERTY_SALE.getValue().equals(billItem.getItemTypeCode())) {

                boolean hasNotAddress = specialAdditions.stream().
                        anyMatch(e -> e.get(PROPERTY_ADDRESS_KEY) == null || StringUtils.isEmpty(e.get(PROPERTY_ADDRESS_KEY).toString()));
                if (!hasNotAddress) {
                    boolean containAddressKeyWord = specialAdditions.stream().map(e -> e.get(PROPERTY_ADDRESS_KEY)).
                            filter(Objects::nonNull).allMatch(e -> {
                                List<String> addressKeyWords = ItemTypeCodeEnum.specialInvoiceAddressKeyWord();
                                for (String word : addressKeyWords) {
                                    if (String.valueOf(e).contains(word)) {
                                        return true;
                                    }
                                }
                                return false;
                            });

                    if (!containAddressKeyWord) {
                        throw new SplitBizException("特殊业务详细地址必须包含“街、路、村、乡、镇、道、巷、号” 关键词之一 ");
                    }
                } else {
                    throw new SplitBizException("不动产业务详细地址不能为空 ");
                }
            }

        }
        if (rule.getTaxInvoiceSource().equals(TaxInvoiceSourceEnum.QD.getValue()) && StringUtils.equals(ItemTypeCodeEnum.PROPERTY_RENT.getValue(),billItem.getItemTypeCode())) {
            List<Map<String, Object>> additions = billItem.getSpecialAdditions();
            Optional.ofNullable(additions)
                    .filter(list -> !list.isEmpty())
                    .map(list -> list.get(0).get("areaUnit"))
                    .map(areaUnitObj -> SpecialAreaUnitEnum.getEnum(String.valueOf(areaUnitObj)))
                    .ifPresent(areaUnitEnum -> billItem.setQuantityUnit(areaUnitEnum.getDescription()));
        }
    }

    public void validateInvoiceItem(BillItem billItem, SplitRule rule) {

        String itemName = billItem.getItemName();
        String itemShortName = billItem.getItemShortName();
        String goodsTaxNo = billItem.getGoodsTaxNo();
        BigDecimal taxRate = billItem.getTaxRate();

        if (org.springframework.util.StringUtils.isEmpty(itemName) && !rule.isIgnoreBillItemName()) {
            throw new SplitBizException("商品名称不能为空");
        }

        if (org.springframework.util.StringUtils.isEmpty(itemShortName) && !rule.isIgnoreBillItemShortName()) {
            throw new SplitBizException("税编简称不能为空");
        }

        if (org.springframework.util.StringUtils.isEmpty(goodsTaxNo) && !rule.isIgnoreBillItemGoodsTaxNo()) {
            throw new SplitBizException("税收分类编码不能为空");
        }

        if (taxRate == null && !rule.isIgnoreBillItemTaxRate()) {
            throw new SplitBizException("税率不能为空");
        }

        if (SpecialGoodsTaxNoEnum.INTANGIBLE_ASSETS_OLD.getValue().equals(goodsTaxNo) &&
                (StringUtils.isBlank(goodsTaxNoMode) || !goodsTaxNoMode.equals("1"))) {
            throw new SplitBizException("税编不合法");
        }

    }

    public void calculateInnerAmount(BillItem billItem, SplitRule rule) {

        formatOuterPrice2Inner(billItem, rule);

        billItem.setAmountWithoutTax(billItem.getAmountWithoutTax().subtract(billItem.getInnerDiscountWithoutTax()).subtract(billItem.getInnerPrepayAmountWithoutTax()));
        if (billItem.getInnerDiscountWithoutTax().compareTo(BigDecimal.ZERO) != 0 || billItem.getInnerPrepayAmountWithoutTax().compareTo(BigDecimal.ZERO) != 0) {
            processPriceOrQuantity(billItem, rule);
        }
        billItem.setTaxAmount(billItem.getTaxAmount().subtract(billItem.getInnerPrepayAmountTax()).subtract(billItem.getInnerDiscountTax()));
        billItem.setAmountWithTax(billItem.getTaxAmount().add(billItem.getAmountWithoutTax()));
        billItem.setDiscountWithoutTax(billItem.getOutterDiscountWithoutTax().add(billItem.getOutterPrepayAmountWithoutTax()));
        billItem.setDiscountWithTax(billItem.getOutterDiscountWithTax().add(billItem.getOutterPrepayAmountWithTax()));
        billItem.setDiscountTax(billItem.getOutterDiscountTax().add(billItem.getOutterPrepayAmountTax()));
        billItem.setInnerDiscountWithoutTax(BigDecimal.ZERO);
        billItem.setInnerDiscountTax(BigDecimal.ZERO);
        billItem.setInnerDiscountWithTax(BigDecimal.ZERO);
        billItem.setInnerPrepayAmountWithTax(BigDecimal.ZERO);
        billItem.setInnerPrepayAmountTax(BigDecimal.ZERO);
        billItem.setInnerPrepayAmountWithoutTax(BigDecimal.ZERO);

    }

    public void reCalculateOriginalTaxAmount(SplitRule rule, BillItem billItem, PriceMethod priceMethod) {

        boolean forceBackCalculateTaxAmount = rule.isForceBackCalculateTaxAmount();
        String taxInvoiceSource = rule.getTaxInvoiceSource();
        boolean isRedItem = billItem.getAmountWithoutTax().compareTo(BigDecimal.ZERO) < 0;
        if(TaxInvoiceSourceEnum.SK.getValue().equals(taxInvoiceSource)){
            if (priceMethod == null) {
                throw new SplitBizException("计价方式必填");
            }
            if(billItem.getQuantity()!=null&&billItem.getQuantity().compareTo(BigDecimal.ZERO)!=0) {
                int scale = billItem.getQuantity().stripTrailingZeros().scale();
                BigDecimal totalAmount = billItem.getAmountWithoutTax();
                if (scale > 6) {
                    BigDecimal recalculatedQuantity = totalAmount.divide(billItem.getUnitPrice(), 6, RoundingMode.HALF_UP);
                    billItem.setQuantity(recalculatedQuantity);
                    logger.info("scale:{},amount:{},salesbillItemId:{}", recalculatedQuantity, totalAmount, billItem.getSalesbillItemId());
                }
            }
        }
        if (TaxInvoiceSourceEnum.QD.getValue().equals(taxInvoiceSource) && forceBackCalculateTaxAmount && !isRedItem) {

            if (priceMethod == null) {
                throw new SplitBizException("计价方式必填");
            }

            if (priceMethod == PriceMethod.WITHOUT_TAX) {

                BigDecimal amountWithoutTax = billItem.getAmountWithoutTax();
                BigDecimal taxRate = billItem.getTaxRate();
                BigDecimal newTaxAmount = amountWithoutTax.multiply(taxRate).setScale(2, RoundingMode.HALF_UP);
                BigDecimal newAmountWithTax = amountWithoutTax.add(newTaxAmount);
                billItem.setTaxAmount(newTaxAmount);
                billItem.setAmountWithTax(newAmountWithTax);

                BigDecimal outerDiscountWithoutTax = billItem.getOutterDiscountWithoutTax();
                BigDecimal newOuterDiscountTax = outerDiscountWithoutTax.multiply(taxRate).setScale(2, RoundingMode.HALF_UP);
                BigDecimal newOuterDiscountWithTax = outerDiscountWithoutTax.add(newOuterDiscountTax);
                billItem.setOutterDiscountTax(newOuterDiscountTax);
                billItem.setOutterDiscountWithTax(newOuterDiscountWithTax);

                BigDecimal outerPrepayAmountWithoutTax = billItem.getOutterPrepayAmountWithoutTax();
                BigDecimal newOuterPrePaymentTax = outerPrepayAmountWithoutTax.multiply(taxRate).setScale(2, RoundingMode.HALF_UP);
                BigDecimal newOuterPrePaymentWithTax = outerPrepayAmountWithoutTax.add(newOuterPrePaymentTax);
                billItem.setOutterPrepayAmountTax(newOuterPrePaymentTax);
                billItem.setOutterPrepayAmountWithTax(newOuterPrePaymentWithTax);

            } else if (priceMethod == PriceMethod.WITH_TAX) {

                BigDecimal taxRate = billItem.getTaxRate();
                BigDecimal newTaxRate = taxRate.add(BigDecimal.ONE);

                BigDecimal amountWithTax = billItem.getAmountWithTax();
                BigDecimal newTaxAmount = amountWithTax.multiply(taxRate).divide(newTaxRate, 2, RoundingMode.HALF_UP);
                BigDecimal newAmountWithoutTax = amountWithTax.subtract(newTaxAmount);

                billItem.setAmountWithoutTax(newAmountWithoutTax);
                billItem.setTaxAmount(newTaxAmount);

                BigDecimal outerDiscountWithTax = billItem.getOutterDiscountWithTax();
                BigDecimal newOuterDiscountWithoutTax = outerDiscountWithTax.divide(newTaxRate, 2, RoundingMode.HALF_UP);
                BigDecimal newOuterDiscountTax = outerDiscountWithTax.subtract(newOuterDiscountWithoutTax);

                billItem.setOutterDiscountWithoutTax(newOuterDiscountWithoutTax);
                billItem.setOutterDiscountTax(newOuterDiscountTax);

                BigDecimal outerPrePaymentWithTax = billItem.getOutterPrepayAmountWithTax();
                BigDecimal newOuterPrePaymentWithoutTax = outerPrePaymentWithTax.divide(newTaxRate, 2, RoundingMode.HALF_UP);
                BigDecimal newOuterPrePaymentTax = outerPrePaymentWithTax.subtract(newOuterPrePaymentWithoutTax);

                billItem.setOutterPrepayAmountWithoutTax(newOuterPrePaymentWithoutTax);
                billItem.setOutterPrepayAmountTax(newOuterPrePaymentTax);

                if (BigDecimal.ZERO.compareTo(newAmountWithoutTax) != 0) {
                    processPriceOrQuantity(billItem, rule);
                }

            }

            /**
             * discount数据覆盖
             */
            billItem.setDiscountWithoutTax(billItem.getOutterDiscountWithoutTax().add(billItem.getOutterPrepayAmountWithoutTax()));
            billItem.setDiscountWithTax(billItem.getOutterDiscountWithTax().add(billItem.getOutterPrepayAmountWithTax()));
            billItem.setDiscountTax(billItem.getOutterDiscountTax().add(billItem.getOutterPrepayAmountTax()));
        }

    }

    /**
     * 20230307关于单价*数量和金额的校验逻辑
     * 无论设备，单价*数量四舍五入后与金额都应该全等
     * 单价数量都为0场景跳过
     * 20231024 数电金额校验优化
     */
    public void validateItemAmount(BillItem billItem, SplitRule rule) {

        BigDecimal unitPrice = billItem.getUnitPrice();
        BigDecimal quantity = billItem.getQuantity();
        boolean enableAdvancedValidation = rule.isEnableAdvancedValidation();
        boolean forceBackCalculateTaxAmount = rule.isForceBackCalculateTaxAmount();
        //String taxInvoiceSource = rule.getTaxInvoiceSource();
        //boolean isSpecialAddition = enableAdvancedValidation && TaxInvoiceSourceEnum.QD.getValue().equals(taxInvoiceSource);
        boolean isSpecialAddition = enableAdvancedValidation;

        if (isSpecialAddition) {

            BigDecimal amountWithoutTax = billItem.getAmountWithoutTax();
            BigDecimal taxAmount = billItem.getTaxAmount();
            BigDecimal taxRate = billItem.getTaxRate();
            boolean isRedItem = amountWithoutTax.compareTo(BigDecimal.ZERO) < 0;
            BigDecimal newTaxAmount = amountWithoutTax.multiply(taxRate).setScale(2, RoundingMode.HALF_UP);

             /**
             * 红字误差1.27蓝字0.06
             */
            //BigDecimal offsetAmount = isRedItem ? BigDecimal.valueOf(1.27) : BigDecimal.valueOf(0.06);
            BigDecimal offsetAmount = BigDecimal.valueOf(0.06);
            if (newTaxAmount.subtract(taxAmount).abs().compareTo(offsetAmount) > 0 && !forceBackCalculateTaxAmount) {
                throw new SplitBizException(String.format("明细id = [%s]  名称 [%s] 不含税金额[%s] * 税率[%s] 与税额[%s] 超过误差[%s] ",
                        billItem.getSalesbillItemId(), billItem.getItemName(), billItem.getAmountWithoutTax(), billItem.getTaxRate(), billItem.getTaxAmount(), offsetAmount));
            }

            BigDecimal outterDiscountWithoutTax = billItem.getDiscountWithoutTax();
            BigDecimal outterDiscountTax = billItem.getOutterDiscountTax();
            BigDecimal newOutterDiscountTax = outterDiscountWithoutTax.multiply(taxRate).setScale(2, RoundingMode.HALF_UP);
            if (newOutterDiscountTax.subtract(outterDiscountTax).abs().compareTo(offsetAmount) > 0 && !forceBackCalculateTaxAmount) {
                throw new SplitBizException(String.format("明细id = [%s]  名称 [%s] 价外折扣不含税金额[%s] * 税率[%s] 与价外折扣税额[%s] 超过误差[%s] ",
                        billItem.getSalesbillItemId(), billItem.getItemName(), billItem.getDiscountWithoutTax(), billItem.getTaxRate(), billItem.getOutterDiscountTax(), offsetAmount));
            }

        }

        //  |不含税单价 * 数量 - 不含税金额 | <=0.01
        if (unitPrice.compareTo(BigDecimal.ZERO) != 0 && quantity.compareTo(BigDecimal.ZERO) != 0) {

            BigDecimal amountWithoutTax = billItem.getAmountWithoutTax();
            BigDecimal newAmountWithoutTax = unitPrice.multiply(quantity).setScale(2, RoundingMode.HALF_UP);
            BigDecimal offsetAmount = isSpecialAddition ? BigDecimal.valueOf(0.01) : BigDecimal.ZERO;

            if (amountWithoutTax.subtract(newAmountWithoutTax).abs().compareTo(offsetAmount) > 0 && !forceBackCalculateTaxAmount) {
                throw new SplitBizException(String.format("明细id = [%s]  名称 [%s] 单价[%s]*数量[%s]取两位小数 与不含税金额[%s]不相等",
                        billItem.getSalesbillItemId(), billItem.getItemName(), billItem.getUnitPrice(), billItem.getQuantity(), billItem.getAmountWithoutTax()));
            }

        }

    }

    public void reCalculateTotalAmount(SplitRule rule, BillInfo billInfo, List<BillItem> items) {

        boolean forceBackCalculateTaxAmount = rule.isForceBackCalculateTaxAmount();
        String taxInvoiceSource = rule.getTaxInvoiceSource();

        if (TaxInvoiceSourceEnum.QD.getValue().equals(taxInvoiceSource) &&
                forceBackCalculateTaxAmount) {

            String salesbillId = billInfo.getSalesbillId();
            String salesbillNo = billInfo.getSalesbillNo();
            Long batchNo = billInfo.getBatchNo();

            logger.info("reCalculateTotalAmount.calculateAmount.salesbillId:{},salesbillNo:{},batchNo:{}",
                    salesbillId, salesbillNo, batchNo);

            BigDecimal totalTaxAmount = BigDecimal.ZERO;
            BigDecimal totalAmountWithoutTax = BigDecimal.ZERO;
            BigDecimal totalAmountWithTax = BigDecimal.ZERO;
            for (BillItem item : items) {
                totalTaxAmount = totalTaxAmount.add(item.getTaxAmount());
                totalAmountWithoutTax = totalAmountWithoutTax.add(item.getAmountWithoutTax());
                totalAmountWithTax = totalAmountWithTax.add(item.getAmountWithTax());
            }

            billInfo.setAmountWithoutTax(totalAmountWithoutTax);
            billInfo.setAmountWithTax(totalAmountWithTax);
            billInfo.setTaxAmount(totalTaxAmount);
        }

    }

    /**
     * 价外折扣优化场景（价外折扣 转 价内折扣）
     *
     * @param billItem
     */
    public void formatOuterPrice2Inner(BillItem billItem, SplitRule splitRule) {

        boolean hideOuterAmount = splitRule.isHideOuterAmount();
        boolean hasOuterDiscount = billItem.getOutterDiscountWithoutTax().compareTo(BigDecimal.ZERO) != 0;
        boolean hasOuterPrepayment = billItem.getOutterPrepayAmountWithoutTax().compareTo(BigDecimal.ZERO) != 0;

        /**
         * 1.金额>0 & 折扣>0 & 金额-折扣<0
         * 2.金额>0 & 折扣<0
         * 3.金额<0 & 折扣>0
         * 4.金额<0 & 折扣<0
         * 5.规则 && 有价外折扣
         */
        if ((hideOuterAmount && hasOuterDiscount) ||
                compareWhenAmountPosAndOutDiscountPosAndSubtractNeg(billItem.getAmountWithoutTax(), billItem.getOutterDiscountWithoutTax()) ||
                compareWhenAmountPosAndOutDiscountNeg(billItem.getAmountWithoutTax(), billItem.getOutterDiscountWithoutTax()) ||
                compareWhenAmountNegAndOutDiscountPos(billItem.getAmountWithoutTax(), billItem.getOutterDiscountWithoutTax()) ||
                compareWhenAmountNegAndOutDiscountNeg(billItem.getAmountWithoutTax(), billItem.getOutterDiscountWithoutTax())) {

            billItem.setInnerDiscountTax(billItem.getInnerDiscountTax().add(billItem.getOutterDiscountTax()));
            billItem.setInnerDiscountWithoutTax(billItem.getInnerDiscountWithoutTax().add(billItem.getOutterDiscountWithoutTax()));
            billItem.setInnerDiscountWithTax(billItem.getInnerDiscountWithTax().add(billItem.getOutterDiscountWithTax()));
            billItem.setOutterDiscountTax(BigDecimal.ZERO);
            billItem.setOutterDiscountWithTax(BigDecimal.ZERO);
            billItem.setOutterDiscountWithoutTax(BigDecimal.ZERO);
        }

        if ((hideOuterAmount && hasOuterPrepayment) ||
                compareWhenAmountPosAndOutDiscountPosAndSubtractNeg(billItem.getAmountWithoutTax(), billItem.getOutterPrepayAmountWithoutTax()) ||
                compareWhenAmountPosAndOutDiscountNeg(billItem.getAmountWithoutTax(), billItem.getOutterPrepayAmountWithoutTax()) ||
                compareWhenAmountNegAndOutDiscountPos(billItem.getAmountWithoutTax(), billItem.getOutterPrepayAmountWithoutTax()) ||
                compareWhenAmountNegAndOutDiscountNeg(billItem.getAmountWithoutTax(), billItem.getOutterPrepayAmountWithoutTax())) {

            billItem.setInnerPrepayAmountTax(billItem.getInnerPrepayAmountTax().add(billItem.getOutterPrepayAmountTax()));
            billItem.setInnerPrepayAmountWithoutTax(billItem.getInnerPrepayAmountWithoutTax().add(billItem.getOutterPrepayAmountWithoutTax()));
            billItem.setInnerPrepayAmountWithTax(billItem.getInnerPrepayAmountWithTax().add(billItem.getOutterPrepayAmountWithTax()));
            billItem.setOutterPrepayAmountWithTax(BigDecimal.ZERO);
            billItem.setOutterPrepayAmountWithoutTax(BigDecimal.ZERO);
            billItem.setOutterPrepayAmountTax(BigDecimal.ZERO);
        }

    }

    private Boolean compareWhenAmountPosAndOutDiscountPosAndSubtractNeg(BigDecimal amount, BigDecimal discount) {
        return amount.compareTo(BigDecimal.ZERO) > 0 &&
                discount.compareTo(BigDecimal.ZERO) > 0 &&
                amount.subtract(discount).compareTo(BigDecimal.ZERO) < 0;
    }

    private Boolean compareWhenAmountPosAndOutDiscountNeg(BigDecimal amount, BigDecimal discount) {
        return amount.compareTo(BigDecimal.ZERO) > 0 &&
                discount.compareTo(BigDecimal.ZERO) < 0;
    }

    private Boolean compareWhenAmountNegAndOutDiscountPos(BigDecimal amount, BigDecimal discount) {
        return amount.compareTo(BigDecimal.ZERO) < 0 &&
                discount.compareTo(BigDecimal.ZERO) > 0;
    }

    private Boolean compareWhenAmountNegAndOutDiscountNeg(BigDecimal amount, BigDecimal discount) {
        return amount.compareTo(BigDecimal.ZERO) < 0 &&
                discount.compareTo(BigDecimal.ZERO) < 0;
    }

    /**
     * 反算必要的数量单价
     *
     * @param billItem
     * @param rule
     */
    public void processPriceOrQuantity(BillItem billItem, SplitRule rule) {

        int unitPriceScale = rule.getUnitPriceScale();

        if (backCalculateUnitPrice(rule, billItem)) {

            BigDecimal newUnitPrice = billItem.getAmountWithoutTax().divide(billItem.getQuantity(), unitPriceScale, RoundingMode.HALF_UP);
            billItem.setUnitPrice(newUnitPrice);
        }

        if (backCalculateQuantity(rule, billItem)) {

            BigDecimal newQuantity;
            BigDecimal newUnitPrice;
            if (backCalculateQuantityRounding(rule, billItem)) {

                newQuantity = billItem.getAmountWithoutTax().divide(billItem.getUnitPrice(), 0, RoundingMode.HALF_UP);
                if (newQuantity.compareTo(BigDecimal.ZERO) == 0) {
                    newQuantity = MIN_QUANTITY;
                }
                newUnitPrice = billItem.getAmountWithoutTax().divide(newQuantity, unitPriceScale, RoundingMode.HALF_UP);
                billItem.setUnitPrice(newUnitPrice);
            } else {
                newQuantity = billItem.getAmountWithoutTax().divide(billItem.getUnitPrice(), 6, RoundingMode.HALF_UP);
                if (newQuantity.compareTo(BigDecimal.ZERO) == 0) {
                    newQuantity = MIN_QUANTITY;
                }
                if (newQuantity.multiply(billItem.getUnitPrice()).setScale(2, RoundingMode.HALF_UP)
                        .compareTo(billItem.getAmountWithoutTax()) != 0) {
                    billItem.setUnitPrice(billItem.getAmountWithoutTax().divide(newQuantity, unitPriceScale, RoundingMode.HALF_UP));
                }
            }

            billItem.setQuantity(newQuantity);
        }

        BigDecimal newAmount = billItem.getUnitPrice().multiply(billItem.getQuantity()).setScale(2, RoundingMode.HALF_UP);
        if (newAmount.compareTo(BigDecimal.ZERO) != 0 && gtErrorAmount(billItem, newAmount)) {
            throw new SplitBizException(String.format("无法计算明细id = [%s] 名称 [%s] 的单价以及数量, 误差超过一分", billItem.getSalesbillItemId(), billItem.getItemName()));
        }

    }

    private boolean backCalculateQuantity(SplitRule rule, BillItem billItem) {
        return !AmountSplitRuleUtils.isQuantity(rule.getAmountSplitRule()) &&
                billItem.getUnitPrice().compareTo(BigDecimal.ZERO) != 0;
    }

    private boolean backCalculateUnitPrice(SplitRule rule, BillItem billItem) {
        return AmountSplitRuleUtils.isQuantity(rule.getAmountSplitRule()) &&
                billItem.getQuantity().compareTo(BigDecimal.ZERO) != 0;
    }

    private boolean backCalculateQuantityRounding(SplitRule rule, BillItem billItem) {
        return AmountSplitRuleUtils.recalculateQuantityRounding(rule.getAmountSplitRule()) &&
                billItem.getUnitPrice().compareTo(BigDecimal.ZERO) != 0;
    }

    private boolean gtErrorAmount(BillItem billItem, BigDecimal amount) {
        return amount.subtract(billItem.getAmountWithoutTax()).abs().compareTo(new BigDecimal(ERROR_AMOUNT)) > 0;
    }

}
