package com.xforceplus.phoenix.split.service;

import com.google.common.collect.ImmutableSet;
import com.xforceplus.phoenix.split.constant.InvoiceItemOrder;
import com.xforceplus.phoenix.split.constant.InvoiceType;
import com.xforceplus.phoenix.split.constant.PreInvoiceTemplateVersionStatus;
import com.xforceplus.phoenix.split.constant.SaleListOptionEnum;
import com.xforceplus.phoenix.split.constant.TaxInvoiceSourceEnum;
import com.xforceplus.phoenix.split.domain.SplitGroupLimit;
import com.xforceplus.phoenix.split.exception.SplitBizException;
import com.xforceplus.phoenix.split.exception.SplitRuleParamException;
import com.xforceplus.phoenix.split.model.BillInfo;
import com.xforceplus.phoenix.split.model.BillItem;
import com.xforceplus.phoenix.split.model.ItemTypeCodeEnum;
import com.xforceplus.phoenix.split.model.SplitRule;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;

import java.math.BigDecimal;
import java.util.Map;

import static com.xforceplus.phoenix.split.constant.SaleListDefaultRule.ELECTRONIC_SALES_LIST_MAX_ROW;
import static com.xforceplus.phoenix.split.constant.SaleListDefaultRule.SALES_LIST_MAX_ROW;

public class SplitRuleUtil {

    public static SplitGroupLimit createSplitGroupLimit(SplitRule rule, String itemTypeCode) {
        SplitGroupLimit splitGroupLimit = new SplitGroupLimit();
        splitGroupLimit.setLimitIsAmountWithTax(rule.isLimitIsAmountWithTax());
        splitGroupLimit.setInvoiceMaxErrorAmount(rule.getInvoiceMaxErrorAmount());
        splitGroupLimit.setLimitAmount(rule.getInvoiceLimit());
        splitGroupLimit.setInvoiceItemOrder(InvoiceItemOrder.fromValue(rule.getItemSort()));
        splitGroupLimit.setLimitLine(getLimitLine(rule, itemTypeCode));
        return splitGroupLimit;
    }

    /**
     * 发票红冲没有限额限制
     * 红字明细 && (原发票号码代码存在 || 红字信息表编号存在)
     * @param rule
     * @param itemTypeCode
     * @param billInfo
     * @param billItem
     * @return
     */
    public static SplitGroupLimit createSplitGroupLimit(SplitRule rule, String itemTypeCode, BillInfo billInfo, BillItem billItem) {

        BigDecimal invoiceLimit = billItem.isRedItem() &&
                (StringUtils.isNotBlank(billInfo.getOriginInvoiceNo()) ||
                        StringUtils.isNotBlank(billInfo.getOriginInvoiceCode()) ||
                        StringUtils.isNotBlank(billInfo.getRedNotificationNo())) ? BigDecimal.valueOf(99999999999.99) : rule.getInvoiceLimit();

        SplitGroupLimit splitGroupLimit = new SplitGroupLimit();
        splitGroupLimit.setLimitIsAmountWithTax(rule.isLimitIsAmountWithTax());
        splitGroupLimit.setInvoiceMaxErrorAmount(rule.getInvoiceMaxErrorAmount());
        splitGroupLimit.setLimitAmount(invoiceLimit);
        splitGroupLimit.setInvoiceItemOrder(InvoiceItemOrder.fromValue(rule.getItemSort()));
        splitGroupLimit.setLimitLine(getLimitLine(rule, itemTypeCode));
        if (rule.isEnableAdvancedValidation()) {
            splitGroupLimit.setInvoiceMaxErrorAmount(BigDecimal.valueOf(1.27));
        }
        if (MapUtils.isNotEmpty(rule.getQuantityLimitMap()) && StringUtils.isNotBlank(billItem.getQuantityUnit()) && rule.getQuantityLimitMap().containsKey(billItem.getQuantityUnit())) {
            splitGroupLimit.setLimitQuantity(rule.getQuantityLimitMap().get(billItem.getQuantityUnit()));
        }
        return splitGroupLimit;
    }

    /**
     * 获取行数限制
     * @param rule
     * @param isCreateSalesList
     * @param itemTypeCode
     * @return
     * @see com.xforceplus.phoenix.split.model.ItemTypeCodeEnum
     */
    public static int getLimitLine(SplitRule rule, boolean isCreateSalesList, String itemTypeCode) {

        if (!isCreateSalesList) {
            return rule.getInvoiceItemMaxRow();
        }

        if (SaleListOptionEnum.DISABLE_LIST_OPTION.getValue().equals(rule.getSaleListOption())) {
            return rule.getInvoiceItemMaxRow();

        } else if (SaleListOptionEnum.ENABLE_LIST_OPTION.getValue().equals(rule.getSaleListOption())) {
            return Math.max(SplitRuleUtil.getSalesListMaxRow(rule), rule.getInvoiceItemMaxRow());

        } else if (SaleListOptionEnum.FORCE_LIST_OPTION.getValue().equals(rule.getSaleListOption())) {
            return SplitRuleUtil.getSalesListMaxRow(rule);

        } else {
            throw new SplitBizException(String.format("value = [%s] for saleListOption is illegal", rule.getSaleListOption()));
        }
    }

    public static int getLimitLine(SplitRule rule, String itemTypeCode) {

        /**
         * 成品油返回发票最大明细
         */
        if (ItemTypeCodeEnum.OIL.getValue().equals(itemTypeCode)) {
            return rule.getInvoiceItemMaxRow();
        }

        if (SaleListOptionEnum.DISABLE_LIST_OPTION.getValue().equals(rule.getSaleListOption())) {
            return rule.getInvoiceItemMaxRow();

        } else if (SaleListOptionEnum.ENABLE_LIST_OPTION.getValue().equals(rule.getSaleListOption())) {
            return Math.max(SplitRuleUtil.getSalesListMaxRow(rule), rule.getInvoiceItemMaxRow());

        } else if (SaleListOptionEnum.FORCE_LIST_OPTION.getValue().equals(rule.getSaleListOption())) {
            return SplitRuleUtil.getSalesListMaxRow(rule);

        } else {
            throw new SplitBizException(String.format("value = [%s] for saleListOption is illegal", rule.getSaleListOption()));
        }
    }

    public static void validateSaleListOption(SplitRule rule) {
        if (!ImmutableSet.of(
                SaleListOptionEnum.DISABLE_LIST_OPTION.getValue(),
                SaleListOptionEnum.ENABLE_LIST_OPTION.getValue(),
                SaleListOptionEnum.FORCE_LIST_OPTION.getValue()).contains(rule.getSaleListOption())) {
            throw new SplitRuleParamException("销货清单参数只能为：0-不启动销货清单,1-启动销货清单,2-强制启用销货清单");
        }
    }

    public static void validateInvoiceItemMaxRow(SplitRule rule) {

        Integer invoiceItemMaxRow = rule.getInvoiceItemMaxRow();
        if (invoiceItemMaxRow == null) {
            throw new SplitRuleParamException("发票明细最大行数不能为空");
        }
        if (invoiceItemMaxRow < 1) {
            throw new SplitRuleParamException("发票明细最大行数必须大于0");
        }

        if (TaxInvoiceSourceEnum.SK.getValue().equals(rule.getTaxInvoiceSource()) &&
                invoiceItemMaxRow > PreInvoiceTemplateVersionStatus.EIGHT.getValue()) {
            throw new SplitRuleParamException("发票明细最大行数不能超过8");
        }

    }

    /**
     * @param rule
     * @return
     */
    public static void validateSalesListMaxRow(SplitRule rule) {

        if (SaleListOptionEnum.DISABLE_LIST_OPTION.getValue().equals(rule.getSaleListOption())) {
            return;
        }

        Integer salesListMaxRow = rule.getSalesListMaxRow();
        if (salesListMaxRow == null) {
            throw new SplitRuleParamException("启用销货清单时,单销货清单最大行数不能为空");
        }

        if (salesListMaxRow < 1) {
            throw new SplitRuleParamException("启用销货清单时，销货清单最大行数必须大于0");
        }
    }

    /**
     * @param rule
     * @return
     */
    public static int getSalesListMaxRow(SplitRule rule) {

        String taxInvoiceSource = rule.getTaxInvoiceSource();
        Integer salesListMaxRow = rule.getSalesListMaxRow();
        if (TaxInvoiceSourceEnum.QD.getValue().equals(taxInvoiceSource)) {
            return salesListMaxRow;
        }

        if (InvoiceType.ELECTRONIC.value().equals(rule.getInvoiceType()) && salesListMaxRow > ELECTRONIC_SALES_LIST_MAX_ROW) {
            salesListMaxRow = ELECTRONIC_SALES_LIST_MAX_ROW;

        } else if (salesListMaxRow > SALES_LIST_MAX_ROW) {
            salesListMaxRow = SALES_LIST_MAX_ROW;
        }

        return salesListMaxRow;
    }

    public static void validateUnitPriceAmountOps(SplitRule rule) {
        try {
            Byte.valueOf(rule.getUnitPriceAmountOps());
        } catch (Exception e) {
            throw new SplitRuleParamException("打印单价数量规则有误");
        }
    }

    /**
     * 校验拆票规则合法性
     * @param rule
     */
    public static void validateSplitRule(SplitRule rule) {

        validateTaxInvoiceSource(rule);

        validateSaleListOption(rule);

        validateInvoiceItemMaxRow(rule);

        validateSalesListMaxRow(rule);

        validateUnitPriceAmountOps(rule);

        validateInvoiceLimit(rule);

        validateQuantityLimit(rule);
    }

    public static void validateTaxInvoiceSource(SplitRule rule) {

        if (!ImmutableSet.of(TaxInvoiceSourceEnum.SK.getValue(), TaxInvoiceSourceEnum.QD.getValue()).contains(rule.getTaxInvoiceSource())) {
            throw new SplitRuleParamException("平台类型参数只能为：sk-税控平台类型,qd-全电平台类型");
        }
    }

    /**
     * 校验拆票限额
     * 全电设置默认值
     * @param rule
     */
    public static void validateInvoiceLimit(SplitRule rule) {

        String taxInvoiceSource = rule.getTaxInvoiceSource();
        BigDecimal invoiceLimit = rule.getInvoiceLimit();

        if (invoiceLimit == null || invoiceLimit.compareTo(BigDecimal.ZERO) <= 0) {

            if (TaxInvoiceSourceEnum.SK.getValue().equals(taxInvoiceSource)) {
                throw new SplitRuleParamException("拆票限额不能为空且必须大于0");

            } else {

                rule.setInvoiceLimit(new BigDecimal("900000000.00"));
            }
        }

    }

    /**
     * 校验请求参数里的quantityLimit是否合法
     */
    public static void validateQuantityLimit(SplitRule rule) {
        final Map<String, BigDecimal> quantityLimitMap = rule.getQuantityLimitMap();
        if (MapUtils.isNotEmpty(quantityLimitMap)) {
            if (!AmountSplitRuleUtils.isUnitPriceAndQuantityInteger(rule.getAmountSplitRule()) && !AmountSplitRuleUtils.isUnitPrice(rule.getAmountSplitRule())) {
                throw new SplitRuleParamException("使用数量限额时，金额拆分规则只能配置为：拆分数量，数量取整（总数量不变）或 拆分数量，数量不取整（总数量不变）");
            }
        }
    }

    public static BigDecimal getErrorTaxAmount(SplitRule splitRule, boolean isRedItem) {
        return getErrorTaxAmountNew(splitRule);
    }


    public static BigDecimal getErrorTaxAmountNew(SplitRule splitRule) {

        String taxInvoiceSource = splitRule.getTaxInvoiceSource();
        if(!StringUtils.equals(taxInvoiceSource,TaxInvoiceSourceEnum.QD.getValue()) && !StringUtils.equals(taxInvoiceSource,TaxInvoiceSourceEnum.SK.getValue())) {
            throw new SplitRuleParamException("找不到误差配置");
        }
        boolean enableAdvancedValidation = splitRule.isEnableAdvancedValidation();

        if (enableAdvancedValidation) {
            //return isRedItem ? BigDecimal.valueOf(1.27) : BigDecimal.valueOf(0.06);
            return new BigDecimal("0.06");
        }

        return BigDecimal.valueOf(0.01);
    }

}
