package com.xforceplus.phoenix.split.service.dataflow;

import com.google.common.collect.Lists;
import com.xforceplus.phoenix.split.constant.InvoiceItemOrder;
import com.xforceplus.phoenix.split.constant.InvoiceType;
import com.xforceplus.phoenix.split.constant.TaxDeviceType;
import com.xforceplus.phoenix.split.domain.ItemGroup;
import com.xforceplus.phoenix.split.domain.SplitGroupLimit;
import com.xforceplus.phoenix.split.model.BillInfo;
import com.xforceplus.phoenix.split.model.BillItem;
import com.xforceplus.phoenix.split.model.ItemTypeCodeEnum;
import com.xforceplus.phoenix.split.model.SplitRule;
import com.xforceplus.phoenix.split.service.dataflow.impl.InvoiceLimitProcessPlugin;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.stream.Collectors;

/**
 * @Author chenlingwei
 * @create 2023/1/5 10:20 AM
 */

@Service
public class SpecialInvoiceService extends InvoiceLimitProcessPlugin {

    protected Integer getAllElectronicSpecialInvoiceItemMaxRow(String itemTypeCode, String invoiceType) {

        /**
         * 不动产销售：1
         * 不动产融资租凭服务：1
         * 建筑服务：1
         * 货物运输：8
         *
         * 20230104更新:都有价外折扣可能
         */

        ItemTypeCodeEnum itemTypeCodeEnum = ItemTypeCodeEnum.fromValue(itemTypeCode);
        if (itemTypeCodeEnum == ItemTypeCodeEnum.BUILDING) {
            return 1;
        }
        else if (itemTypeCodeEnum == ItemTypeCodeEnum.CARGO_TRANSPORT) {
            if (InvoiceType.isNormalOrSpecialInvoice(invoiceType)) {
                return 1;
            } else {
                return 2000;
            }
        }
        else if (itemTypeCodeEnum == ItemTypeCodeEnum.PROPERTY_SALE) {
            return 1;
        } else if (itemTypeCodeEnum == ItemTypeCodeEnum.PROPERTY_RENT) {
            return 2000;
        } else if (itemTypeCodeEnum == ItemTypeCodeEnum.SCRAPPED_PRODUCT_ACQUISITION) {
            return 2000;
        }

        return null;
    }

    /**
     * 处理特殊票种成票
     * @param specialBillItems
     * @return
     */
    public List<ItemGroup> assembleItems(SplitRule rule, ItemGroup itemGroup, BillInfo billInfo, List<BillItem> specialBillItems) {

        Map<String, List<BillItem>> itemTypeCodeBillItemMap = specialBillItems.stream().collect(Collectors.groupingBy(BillItem::getItemTypeCode));

        /**
         * 特殊票种根据itemTypeCode区分
         * 单个明细成票，判断是否行数超限以及税差超限
         */
        List<ItemGroup> results = new ArrayList<>();
        for (Map.Entry<String, List<BillItem>> entry : itemTypeCodeBillItemMap.entrySet()) {
            String itemTypeCode = entry.getKey();
            List<BillItem> billItems = entry.getValue();

            SplitGroupLimit splitGroupLimit = new SplitGroupLimit();
            splitGroupLimit.setLimitIsAmountWithTax(rule.isLimitIsAmountWithTax());
            splitGroupLimit.setInvoiceMaxErrorAmount(rule.getInvoiceMaxErrorAmount());
            splitGroupLimit.setLimitAmount(BigDecimal.valueOf(99999999999.99));
            splitGroupLimit.setInvoiceItemOrder(InvoiceItemOrder.fromValue(rule.getItemSort()));
            splitGroupLimit.setLimitLine(getAllElectronicSpecialInvoiceItemMaxRow(itemTypeCode, billInfo.getInvoiceType()));

            while (!billItems.isEmpty()) {
                List<BillItem> processedItemList = splitByAmountAndLineLimit(billItems, itemTypeCode, splitGroupLimit);
                ItemGroup processedItemGroup = new ItemGroup(itemGroup.getParentGroupFlag());
                processedItemGroup.setInvoiceOfGroupKey(UUID.randomUUID().toString());
                processedItemGroup.setBillItems(processedItemList);
                results.add(processedItemGroup);
            }

        }

        return results;
    }

    /**
     * 为后续特殊票种提前扩展
     * @param billItems
     * @param itemTypeCode    特殊票种类型
     * @param splitGroupLimit
     * @return
     */
    protected List<BillItem> splitByAmountAndLineLimit(List<BillItem> billItems, String itemTypeCode, SplitGroupLimit splitGroupLimit) {

        BigDecimal amount = BigDecimal.ZERO;
        int lineNum = 0;
        List<BillItem> splitGroup = Lists.newLinkedList();
        BigDecimal totalErrorAmount = BigDecimal.ZERO;
        BigDecimal totalErrorAmountAbs = BigDecimal.ZERO;

        /**
         * 开票商税额差配置
         */
        BigDecimal basicInvoiceMaxErrorAmount = BigDecimal.valueOf(1.26);

        /**
         * 项目税额差配置
         */
        BigDecimal businessInvoiceMaxErrorAmount = splitGroupLimit.getInvoiceMaxErrorAmount();

        for (Iterator<BillItem> it = billItems.iterator(); it.hasNext(); ) {
            BillItem item = it.next();

            if (item.getDeductions().compareTo(BigDecimal.ZERO) > 0) {
                if (splitGroup.size() > 0) {
                    continue;
                }
                splitGroup.add(item);
                it.remove();
                return splitGroup;
            }

            // 金额限制
            boolean isAmountGtLimit = amount.add(actualAmount(item, splitGroupLimit.isLimitIsAmountWithTax())).compareTo(splitGroupLimit.getLimitAmount()) > 0;
            // 行数限制
            boolean isLineGtLimit = lineNum + this.needLineNum(item) > splitGroupLimit.getLimitLine();
            // 税额误差不超过1.27
            BigDecimal currentItemErrorAmount = errorAmountCalculation(item, TaxDeviceType.HX_SINGL);
            BigDecimal currentItemErrorAmountAbs = errorAmountCalculation(item, TaxDeviceType.HX_SINGL);

            boolean rateFor1dot27 = totalErrorAmountCalculation(totalErrorAmount, currentItemErrorAmount, basicInvoiceMaxErrorAmount, TaxDeviceType.HX_SINGL);
            boolean rateForBusinessRule = totalErrorAmountCalculationForBusiness(totalErrorAmountAbs, currentItemErrorAmountAbs, businessInvoiceMaxErrorAmount);
            //当行数超限 或 税额误差超过1.27
            if (isLineGtLimit || rateFor1dot27 || rateForBusinessRule) {
                return splitGroup;
            }
            if (!isAmountGtLimit) {
                amount = amount.add(actualAmount(item, splitGroupLimit.isLimitIsAmountWithTax()));
                lineNum += this.needLineNum(item);
                totalErrorAmount = totalErrorAmount.add(currentItemErrorAmount);
                totalErrorAmountAbs = totalErrorAmountAbs.add(currentItemErrorAmountAbs);

                splitGroup.add(item);
                it.remove();
            }

        }
        return splitGroup;
    }

    /**
     * 特殊票种 价外折扣不额外算作一条 可以合并成一张票
     * @param item 单据明细
     * @return
     */
    @Override
    public int needLineNum(BillItem item) {
        return 1;
    }

}
