package com.xforceplus.phoenix.split.service.dataflow.impl.invoiceLimit;

import com.google.common.collect.Lists;
import com.xforceplus.phoenix.split.config.SplitProperties;
import com.xforceplus.phoenix.split.constant.TaxDeviceType;
import com.xforceplus.phoenix.split.domain.SplitGroupLimit;
import com.xforceplus.phoenix.split.model.BillItem;
import com.xforceplus.phoenix.split.model.SplitRule;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;

/**
 * 按照明细顺序，组合明细成票
 */
@Component
public class ItemOrderCombineItemService extends AbstractCombineItemService implements CombineItemService {

    public ItemOrderCombineItemService(SplitProperties splitProperties) {
        super(splitProperties);
    }

    /**
     * 组合明细成票，需要满足
     * 1、金额在限额内
     * 2、行数在限制内
     * 3、数量在限制内（配置数量限额时）
     * 4、票面总误差不超过限制
     * 5、票面税额误差在业务单配置限制内
     * 旧逻辑，迁移重构
     */
    @Override
    public List<BillItem> combineItem(List<BillItem> billItems, SplitGroupLimit splitGroupLimit, TaxDeviceType taxDeviceType, SplitRule splitRule) {
        BigDecimal amount = BigDecimal.ZERO;
        int lineNum = 0;
        BigDecimal quantity = BigDecimal.ZERO;
        BigDecimal totalErrorAmount = BigDecimal.ZERO;
        BigDecimal totalErrorAmountAbs = BigDecimal.ZERO;

        //开票商税额差配置
        BigDecimal basicInvoiceMaxErrorAmount = this.getMaxErrorAmountByTaxDeviceType(taxDeviceType, splitRule);

        // 项目税额差配置
        BigDecimal businessInvoiceMaxErrorAmount = splitGroupLimit.getInvoiceMaxErrorAmount();

        List<BillItem> splitGroup = Lists.newLinkedList();

        for (Iterator<BillItem> it = billItems.iterator(); it.hasNext(); ) {
            BillItem item = it.next();

            // 差额征税时，单条开票
            if (item.getDeductions().compareTo(BigDecimal.ZERO) > 0) {
                if (CollectionUtils.isNotEmpty(splitGroup)) {
                    continue;
                }
                splitGroup.add(item);
                it.remove();
                return splitGroup;
            }

            // 金额限制
            boolean isAmountGtLimit = amount.add(super.actualAmount(item, splitGroupLimit.isLimitIsAmountWithTax())).compareTo(splitGroupLimit.getLimitAmount()) > 0;
            // 行数限制
            boolean isLineGtLimit = lineNum + this.actualLineNum(item) > splitGroupLimit.getLimitLine();
            //数量限制
            boolean isQuantityGtLimit = Objects.nonNull(splitGroupLimit.getLimitQuantity()) && quantity.add(item.getQuantity()).compareTo(splitGroupLimit.getLimitQuantity()) > 0;
            // 税额误差不超过1.27
            BigDecimal currentItemErrorAmount = this.errorAmountCalculation(item, taxDeviceType);
            BigDecimal currentItemErrorAmountAbs = this.errorAmountCalculation(item, TaxDeviceType.HX_SINGL);

            boolean rateFor1dot27 = this.totalErrorAmountCalculation(totalErrorAmount, currentItemErrorAmount, basicInvoiceMaxErrorAmount, taxDeviceType);
            boolean rateForBusinessRule = this.totalErrorAmountCalculationForBusiness(totalErrorAmountAbs, currentItemErrorAmountAbs, businessInvoiceMaxErrorAmount);

            //return 还是 continue，是核心差异逻辑
            //当任意不满足时，返回
            if (isAmountGtLimit || isQuantityGtLimit || isLineGtLimit || rateFor1dot27 || rateForBusinessRule) {
                return splitGroup;
            }

            amount = amount.add(super.actualAmount(item, splitGroupLimit.isLimitIsAmountWithTax()));
            lineNum += this.actualLineNum(item);
            quantity = Objects.nonNull(splitGroupLimit.getLimitQuantity()) ? quantity.add(item.getQuantity()) : BigDecimal.ZERO;
            totalErrorAmount = totalErrorAmount.add(currentItemErrorAmount);
            totalErrorAmountAbs = totalErrorAmountAbs.add(currentItemErrorAmountAbs);

            splitGroup.add(item);
            it.remove();
        }
        return splitGroup;
    }
}
