/*
 * Copyright (c)  2015~2025, Xforceplus
 * All rights reserved.
 * Project: bill-split-service
 * ClassName: InvoiceLimit
 * Date: 2022-03-18 15:54:19
 * Author: zhouxin
 */
package com.xforceplus.phoenix.split.domain;

/**
 * @Description:
 * @CopyRight: 上海云砺信息科技有限公司
 * @Author: zhouxin
 * @Email: yf_zhouxin@xforceplus.com
 * @Date: 2022/3/18 15:54:19
 */

import com.google.common.base.Objects;
import com.xforceplus.phoenix.split.model.BillItem;
import java.math.BigDecimal;
import java.util.List;

/**
 * 发票成票限制
 */
public class PreInvoiceLimit {

    private final BigDecimal ERROR_AMOUNT = new BigDecimal("1.27");

    /**
     * 额度限制
     */
    private BigDecimal limitAmount;

    /**
     * 额度限制是否是含税金额
     */
    private boolean limitIsAmountWithTax;


    /**
     * 行数限制
     */
    private int limitLine;


    private BigDecimal currentTotalAmount;

    private BigDecimal currentTotalError;

    private int currentLine;


    public PreInvoiceLimit(BigDecimal limitAmount, boolean limitIsAmountWithTax, int limitLine) {
        this.limitAmount = limitAmount;
        this.limitIsAmountWithTax = limitIsAmountWithTax;
        this.limitLine = limitLine;

        this.currentTotalAmount = BigDecimal.ZERO;
        this.currentTotalError = BigDecimal.ZERO;
        this.currentLine = 0;
    }

    public PreInvoiceLimit(SplitGroupLimit splitGroupLimit) {
        this.limitAmount = splitGroupLimit.getLimitAmount();
        this.limitIsAmountWithTax = splitGroupLimit.isLimitIsAmountWithTax();
        this.limitLine = splitGroupLimit.getLimitLine();

        this.currentTotalAmount = BigDecimal.ZERO;
        this.currentTotalError = BigDecimal.ZERO;
        this.currentLine = 0;
    }

    /**
     * 是否还能添加明细 当前金额小于限额 & 当前行数小于限制行数 & 当前误差累计小于等于误差限制 则能加
     */
    boolean canAdd() {
        return currentTotalAmount.compareTo(limitAmount) <= 0 && currentLine < limitLine
            && currentTotalError.compareTo(ERROR_AMOUNT) <= 0;
    }

    public boolean canAddItem(BillItem billItem) {
        BigDecimal amount = getBillItemAmount(billItem);
        BigDecimal errorAmount = getBillItemErrorAmount(billItem);
        int line = getBillItemLine(billItem);

        return currentTotalAmount.add(amount)
                                 .compareTo(limitAmount) <= 0 &&
            (currentLine + line) <= limitLine &&
            currentTotalError.add(errorAmount)
                             .compareTo(this.ERROR_AMOUNT) <= 0;
    }

    public void addItem(BillItem billItem) {
        BigDecimal amount = getBillItemAmount(billItem);
        BigDecimal errorAmount = getBillItemErrorAmount(billItem);
        int line = getBillItemLine(billItem);

        currentTotalAmount = currentTotalAmount.add(amount);
        currentLine += line;
        currentTotalError = currentTotalError.add(errorAmount);
    }


    /**
     * 是否能加多条明细
     */
    public boolean canAddInvoice(List<BillItem> billItems) {

        BigDecimal totalAmount = BigDecimal.ZERO;
        BigDecimal totalErrorAmount = BigDecimal.ZERO;
        int totalLine = 0;
        for (BillItem billItem : billItems) {
            BigDecimal amount = getBillItemAmount(billItem);
            BigDecimal errorAmount = getBillItemErrorAmount(billItem);
            int line = getBillItemLine(billItem);
            totalAmount = totalAmount.add(amount);
            totalErrorAmount = totalErrorAmount.add(errorAmount);
            totalLine += line;

            if (!(currentTotalAmount.add(totalAmount)
                                    .compareTo(limitAmount) <= 0 &&
                (currentLine + totalLine) <= limitLine &&
                currentTotalError.add(totalErrorAmount)
                                 .compareTo(this.ERROR_AMOUNT) <= 0)) {
                return false;
            }

        }

        return true;
    }

    /**
     * 加入多条明细
     */
    public void addInvoice(List<BillItem> billItems) {
        billItems.forEach(this::addItem);
    }

    private int getBillItemLine(BillItem billItem) {
        if (billItem.getDiscountWithoutTax()
                    .compareTo(BigDecimal.ZERO) > 0) {
            return 2;
        }
        return 1;
    }

    private BigDecimal getBillItemErrorAmount(BillItem billItem) {
        return (billItem.getTaxAmount()
                        .subtract(billItem.getDiscountTax())
                        .subtract(
                            (billItem.getAmountWithoutTax()
                                     .subtract(billItem.getDiscountWithoutTax())).multiply(billItem.getTaxRate())
                        )
                        .abs());
    }

    private BigDecimal getBillItemAmount(BillItem billItem) {
        BigDecimal result;

        if (limitIsAmountWithTax) {
            result = billItem.getAmountWithTax()
                             .subtract(billItem.getDiscountWithTax());
        } else {
            result = billItem.getAmountWithoutTax()
                             .subtract(billItem.getDiscountWithoutTax());
        }

        return result;
    }


    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        PreInvoiceLimit that = (PreInvoiceLimit) o;
        return currentLine == that.currentLine &&
            currentTotalError.compareTo(that.currentTotalError) == 0 &&
            currentTotalAmount.compareTo(that.currentTotalAmount) == 0;
    }

    @Override
    public int hashCode() {
        return Objects.hashCode(currentTotalAmount, currentTotalError, currentLine);
    }
}
