package com.xforceplus.phoenix.split.service;

import com.google.common.collect.Sets;
import com.xforceplus.phoenix.split.constant.InvoiceType;
import com.xforceplus.phoenix.split.constant.SaleListOptionEnum;
import com.xforceplus.phoenix.split.constant.TaxInvoiceSourceEnum;
import com.xforceplus.phoenix.split.exception.SplitBizException;
import com.xforceplus.phoenix.split.model.BillInfo;
import com.xforceplus.phoenix.split.model.SplitRule;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.math.BigDecimal;
import java.util.Set;

import static com.xforceplus.phoenix.split.constant.SaleListDefaultRule.SALES_LIST_MAX_ROW;

/**
 * @Author chenlingwei
 * @create 8/17/22 7:37 PM
 */

@Service
public class SplitRuleService {

    @Value("${split.invoice.count.limit.each:1000}")
    private int splitInvoiceCount = 1000;

    @Value("#{'${config.remark.concat.comma.tenantIds}'.split(',')}")
    private Set<String> remarkCommaTenantIds;

    @Value("#{'${config.remark.remove.duplicate.tenantIds}'.split(',')}")
    private Set<String> remarkRemovalDuplicateTenantIds;

    @Value("#{'${config.ignore.item.tax-code.tenantIds}'.split(',')}")
    private Set<String> ignoreItemTaxCodeTenantIds = Sets.newHashSet("111");

    @Value("#{'${config.ignore.item.tax-code.appIds}'.split(',')}")
    private Set<String> ignoreItemTaxCodeAppIds = Sets.newHashSet("20");

    @Value("#{'${config.remove.item-name-illegal-character.tenantIds}'.split(',')}")
    private Set<String> removeItemNameIllegalCharacter = Sets.newHashSet("111");

    // @Value("#{'${config.middleware.appIds}'.split(',')}")
    // private Set<String> splitMiddlewareAppIds = Sets.newHashSet("20");

    /**
     * 拆票前处理
     * @param rule
     * @param billInfo
     */
    public void preHandle(Long tenantId, long appId, BillInfo billInfo, SplitRule rule) {

        initializeSplitRule(tenantId, appId, rule, billInfo);

        SplitRuleUtil.validateSplitRule(rule);

        validateSplitCount(billInfo, rule);

    }

    /**
     * 转换规则中的值
     * 20220909 全电类型票面最大行、销货清单参数 优先使用规则中的值，其次是默认值
     * @param rule
     */
    private void initializeSplitRule(Long tenantId, long appId, SplitRule rule, BillInfo billInfo) {

        //国税来源默认税控
        this.initializeTaxInvoiceSource(rule);

        initializeInvoiceMaxRow(rule, billInfo);

        initializeSaleListOption(rule, billInfo);

        //商品名称分隔符类型 0-空格 1-换行 2-逗号 默认为 2
        initializeRemarkSeparator(tenantId, rule);

        initializeBillItemValidation(tenantId, appId, rule);

        initializeBillItemIllegalCharacterRemoval(tenantId, rule);

        initializeSplitAsyncMiddleWare(appId, rule);
    }

    public void initializeSplitAsyncMiddleWare(long appId, SplitRule rule) {

    }

    /**
     * 初始化租户级商品名称非法字符去除配置
     * @param tenantId
     * @param rule
     */
    public void initializeBillItemIllegalCharacterRemoval(Long tenantId, SplitRule rule) {

        if (removeItemNameIllegalCharacter.contains(String.valueOf(tenantId))) {
            rule.setRemoveIllegalItemNameCharacter(true);
        }
    }

    /**
     * 初始化明细必填项校验规则
     * @param tenantId
     * @param appId
     * @param rule
     */
    public void initializeBillItemValidation(Long tenantId, long appId, SplitRule rule) {

        if (ignoreItemTaxCodeTenantIds.contains(String.valueOf(tenantId)) || ignoreItemTaxCodeAppIds.contains(String.valueOf(appId))) {
            rule.setIgnoreBillItemShortName(true);
            rule.setIgnoreBillItemName(true);
            rule.setIgnoreBillItemTaxRate(true);
            rule.setIgnoreBillItemGoodsTaxNo(true);
        }

    }

    /**
     * @param tenantId
     * @param rule
     */
    public void initializeRemarkSeparator(Long tenantId, SplitRule rule) {

        if (!CollectionUtils.isEmpty(remarkCommaTenantIds) &&
                remarkCommaTenantIds.contains(String.valueOf(tenantId))) {

            rule.setCargoNameSeparatorType(2);
        }

        if (!CollectionUtils.isEmpty(remarkRemovalDuplicateTenantIds) &&
                remarkRemovalDuplicateTenantIds.contains(String.valueOf(tenantId))) {

            rule.setRemoveRemarkItemDuplicate(true);

        }

    }

    /**
     * 全电场景发票最大明细行默认2000
     * 全电纸票沿用税控规则
     * @param rule
     */
    public void initializeInvoiceMaxRow(SplitRule rule, BillInfo billInfo) {

        String taxInvoiceSource = rule.getTaxInvoiceSource();
        Integer invoiceItemMaxRow = rule.getInvoiceItemMaxRow();

        //数电，发票类型非 sz、cz、c、s 成票最大明细行默认不传为 2000
        if (TaxInvoiceSourceEnum.QD.getValue().equals(taxInvoiceSource) &&
                !InvoiceType.isAllElectronicNormalInvoice(billInfo.getInvoiceType())) {

            rule.setInvoiceItemMaxRow(
                    invoiceItemMaxRow == null || invoiceItemMaxRow == 0 ? SALES_LIST_MAX_ROW : invoiceItemMaxRow);

        }

    }

    /**
     * 平台类型空值自动转换为税控类型
     * @param rule
     */
    public void initializeTaxInvoiceSource(SplitRule rule) {

        String taxInvoiceSource = rule.getTaxInvoiceSource();

        rule.setTaxInvoiceSource(StringUtils.isEmpty(taxInvoiceSource) ? "sk" : taxInvoiceSource);
    }

    /**
     * 默认销货清单参数
     * 全电纸票沿用税控规则
     * @param rule
     */
    public void initializeSaleListOption(SplitRule rule, BillInfo billInfo) {

        String taxInvoiceSource = rule.getTaxInvoiceSource();
        Integer invoiceItemMaxRow = rule.getInvoiceItemMaxRow();
        String saleListOption = rule.getSaleListOption();
        Integer salesListMaxRow = rule.getSalesListMaxRow();

        if (TaxInvoiceSourceEnum.QD.getValue().equals(taxInvoiceSource) &&
                !InvoiceType.isAllElectronicNormalInvoice(billInfo.getInvoiceType())) {

            /**
             * 默认生成销货清单
             * 存在场景：实际业务单明细超过配置的发票最大行数，则生成多张预制发票
             */
            rule.setSaleListOption(StringUtils.isEmpty(saleListOption) ?
                    SaleListOptionEnum.ENABLE_LIST_OPTION.getValue() : saleListOption);

            rule.setSalesListMaxRow(salesListMaxRow == null || salesListMaxRow == 0 ? invoiceItemMaxRow : salesListMaxRow);
        }
    }

    /**
     * 校验拆票拆完后张数
     * @param billInfo
     * @param rule
     */
    public void validateSplitCount(BillInfo billInfo, SplitRule rule) {

        boolean limitIsAmountWithTax = rule.isLimitIsAmountWithTax();

        BigDecimal totalSplitAmount = limitIsAmountWithTax ? billInfo.getAmountWithTax() : billInfo.getAmountWithoutTax();

        int splitCount = totalSplitAmount.divide(rule.getInvoiceLimit(), 0, BigDecimal.ROUND_HALF_UP).intValue();

        int splitCountAbs = splitCount < 0 ? -splitCount : splitCount;
        if (splitCountAbs > splitInvoiceCount) {
            throw new SplitBizException(String.format("预估单次拆分预制发票数量达到[%s]超过限制数量[%s]", splitCountAbs, splitInvoiceCount));
        }
    }
}
