package com.xforceplus.phoenix.split.service.dataflow;

import com.google.common.collect.ListMultimap;
import com.google.common.collect.MultimapBuilder;
import com.xforceplus.phoenix.split.constant.TaxDeviceType;
import com.xforceplus.phoenix.split.domain.ItemGroup;
import com.xforceplus.phoenix.split.domain.RuleInfo;
import com.xforceplus.phoenix.split.model.BillInfo;
import com.xforceplus.phoenix.split.model.BillItem;
import com.xforceplus.phoenix.split.model.SplitRule;

import java.util.LinkedList;
import java.util.List;

public abstract class AbstractBillItemGroupPlugin implements DataProcessPlugin {

    public List<ItemGroup> groupBillItemByRule(List<ItemGroup> itemGroups, SplitRule rule) {
        if (support(rule, itemGroups)) {
            return groupBillItem(itemGroups, rule);
        }
        return itemGroups;
    }

    /**
     * 分组明细
     * @param itemGroups
     * @return
     */
    public List<ItemGroup> groupBillItem(List<ItemGroup> itemGroups, SplitRule rule) {
        List<ItemGroup> result = new LinkedList<>();
        for (ItemGroup itemGroup : itemGroups) {
            List<BillItem> billItems = itemGroup.getBillItems();

            ListMultimap<String, BillItem> group = MultimapBuilder.linkedHashKeys().linkedListValues().build();
            billItems.forEach(billItem -> {
                String key = generateKey(billItem, rule);
                group.put(key, billItem);
            });
            group.keySet().forEach(key -> {
                ItemGroup newItemGroup = new ItemGroup();
                newItemGroup.setBillItems(group.get(key));
                result.add(newItemGroup);
            });
        }
        return result;
    }

    /**
     * 创建分组key
     * @param billItem
     * @param rule
     * @return
     */
    protected abstract String generateKey(BillItem billItem, SplitRule rule);

    /**
     * 根据规则判断是否需要分组
     * @param rule
     * @return
     */
    protected abstract boolean support(SplitRule rule, List<ItemGroup> itemGroups);

    @Override
    public List<ItemGroup> processData(List<ItemGroup> itemGroups, BillInfo billInfo, RuleInfo rule) {
        return groupBillItemByRule(itemGroups, rule.getSplitRule());
    }

    @Override
    public List<ItemGroup> processData(List<ItemGroup> itemGroups, BillInfo billInfo, RuleInfo ruleInfo, TaxDeviceType taxDeviceType) {
        return processData(itemGroups, billInfo, ruleInfo);
    }
}
