package com.xforceplus.phoenix.split.service.dataflow.impl;

import com.google.common.base.Stopwatch;
import com.google.common.collect.Lists;
import com.xforceplus.phoenix.split.constant.TaxDeviceType;
import com.xforceplus.phoenix.split.domain.ItemGroup;
import com.xforceplus.phoenix.split.domain.RuleInfo;
import com.xforceplus.phoenix.split.exception.SplitBizException;
import com.xforceplus.phoenix.split.model.BillInfo;
import com.xforceplus.phoenix.split.model.BillItem;
import com.xforceplus.phoenix.split.model.PriceMethod;
import com.xforceplus.phoenix.split.service.BasicDataProcessService;
import com.xforceplus.phoenix.split.service.dataflow.DataProcessPlugin;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.List;

/**
 * 满足业务拆票之外的开票客户端基础数据处理过程
 * ps:明细单价精度不能超过19位
 * @Author chenlingwei
 * @create 6/1/22 2:24 PM
 */

@Service
public class BasicPlugin implements DataProcessPlugin {

    private static final Logger log = LoggerFactory.getLogger(BasicPlugin.class);

    @Autowired
    private BasicDataProcessService basicDataProcessService;

    /**
     * 调整单价
     * 调整数量
     * 调整全额折扣
     * @param itemGroups
     * @param billInfo
     * @param rule
     * @return
     */
    @Override
    public List<ItemGroup> processData(List<ItemGroup> itemGroups, BillInfo billInfo, RuleInfo rule) {

        BigDecimal discountLimit = rule.getSplitRule().getDiscountLimit();
        List<ItemGroup> fullDiscountItemGroups = Lists.newArrayListWithCapacity(itemGroups.size());

        for (ItemGroup itemGroup : itemGroups) {

            List<BillItem> billItems = itemGroup.getBillItems();

            billItems.parallelStream().forEach(e -> basicDataProcessService.adjustBasicUnitPrice(e, rule.getSplitRule()));

            billItems.parallelStream().forEach(e -> basicDataProcessService.adjustBasicQuantityPrecision(e, rule.getSplitRule()));

            BigDecimal invoiceTotalAmount = BigDecimal.ZERO;
            BigDecimal invoiceDiscountAmount = BigDecimal.ZERO;
            for (BillItem item : itemGroup.getBillItems()) {
                if (rule.getSplitRule().getPriceMethod() == PriceMethod.WITHOUT_TAX) {
                    invoiceTotalAmount = invoiceTotalAmount.add(item.getAmountWithoutTax());
                    invoiceDiscountAmount = invoiceDiscountAmount.add(item.getDiscountWithoutTax());
                } else {
                    invoiceTotalAmount = invoiceTotalAmount.add(item.getAmountWithTax());
                    invoiceDiscountAmount = invoiceDiscountAmount.add(item.getDiscountWithTax());
                }
            }

            /**
             * 全额折扣后续处理
             */
            if (invoiceTotalAmount.compareTo(invoiceDiscountAmount) == 0) {
                fullDiscountItemGroups.add(itemGroup);
            }

            BigDecimal invoiceDiscountRate = invoiceDiscountAmount.divide(invoiceTotalAmount, 2, RoundingMode.HALF_UP);
            if (discountLimit != null && invoiceDiscountRate.compareTo(discountLimit) > 0 && CollectionUtils.isEmpty(fullDiscountItemGroups)) {
                throw new SplitBizException(String.format("超过%s折扣比例限制!", discountLimit));
            }

            /**
             * 总明细 - 全额折扣明细
             */
            int eachInvoiceFullDiscountCount = (int) billItems.stream().filter(e -> e.getAmountWithoutTax().compareTo(e.getOutterDiscountWithoutTax()) == 0).count();
            itemGroup.setContainFullDiscountCapacity(billItems.size() - eachInvoiceFullDiscountCount - 1);
        }

        /**
         * 预知发票明细置换
         * 置换不成功则继续报错
         */
        if (!CollectionUtils.isEmpty(fullDiscountItemGroups)) {

            Stopwatch stopwatch = Stopwatch.createStarted();
            itemGroups = basicDataProcessService.adjustInvoiceItemOrder(itemGroups, fullDiscountItemGroups);
            log.info("BasicPlugin.processData adjustInvoiceItemOrder elapsed time = {} ms, fullDiscountItemGroups size:{}",
                    stopwatch.elapsed().toMillis(), fullDiscountItemGroups.size());
        }

        return itemGroups;

    }

    @Override
    public List<ItemGroup> processData(List<ItemGroup> itemGroups, BillInfo billInfo, RuleInfo ruleInfo, TaxDeviceType taxDeviceType) {

        return processData(itemGroups, billInfo, ruleInfo);
    }
}
