package com.xforceplus.phoenix.split.service.dataflow.impl;

import com.xforceplus.phoenix.split.constant.TaxDeviceType;
import com.xforceplus.phoenix.split.domain.ItemGroup;
import com.xforceplus.phoenix.split.domain.RuleInfo;
import com.xforceplus.phoenix.split.exception.SplitBizException;
import com.xforceplus.phoenix.split.model.BillInfo;
import com.xforceplus.phoenix.split.model.BillItem;
import com.xforceplus.phoenix.split.model.FieldOfObj;
import com.xforceplus.phoenix.split.model.RemarkFieldMetadata;
import com.xforceplus.phoenix.split.service.dataflow.DataProcessPlugin;
import com.xforceplus.phoenix.split.util.CommonTools;
import com.xforceplus.phoenix.split.util.FieldUtils;
import org.apache.ibatis.reflection.ReflectionException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import static com.xforceplus.phoenix.split.constant.RemarkConstant.COMMA;
import static com.xforceplus.phoenix.split.constant.RemarkConstant.SPACE;
import static com.xforceplus.phoenix.split.constant.RemarkConstant.TAB;

/**
 * 明细名称处理，可根据拆票规则配置字段拼接名称
 */
@Component
public class FieldProcessPlugin implements DataProcessPlugin {

    private Logger logger = LoggerFactory.getLogger(FieldProcessPlugin.class);

    @Value("${cargoNameLengthLimit:100}")
    private int cargoNameLengthLimit;

    @Value("${cargoNameLengthLimit:40}")
    private int itemSpecNameLengthLimit;

    private static final Charset GBK = Charset.forName("GBK");

    @Override
    public List<ItemGroup> processData(List<ItemGroup> itemGroups, BillInfo billInfo, RuleInfo ruleInfo) {
        List<RemarkFieldMetadata> cargoNameFieldMetadatas = ruleInfo.getSplitRule().getCargoNameFiledMetadatas();
        List<RemarkFieldMetadata> itemSpecFieldMetadatas = ruleInfo.getSplitRule().getItemSpecFiledMetadatas();
        Integer cargoNameSeparatorType = ruleInfo.getSplitRule().getCargoNameSeparatorType();

        if (CollectionUtils.isEmpty(cargoNameFieldMetadatas) && CollectionUtils.isEmpty(itemSpecFieldMetadatas)) {
            return itemGroups;
        }

        int itemSpecNameLengthLimit = ruleInfo.getSplitRule().getItemSpecNameLength() == null ?
                this.itemSpecNameLengthLimit :
                Math.min(ruleInfo.getSplitRule().getItemSpecNameLength(), this.itemSpecNameLengthLimit);

        for (ItemGroup itemGroup : itemGroups) {
            setValue(billInfo, cargoNameFieldMetadatas, itemSpecFieldMetadatas, itemSpecNameLengthLimit, itemGroup, cargoNameSeparatorType);
        }

        return itemGroups;
    }

    @Override
    public List<ItemGroup> processData(List<ItemGroup> itemGroups, BillInfo billInfo, RuleInfo ruleInfo, TaxDeviceType taxDeviceType) {
        return processData(itemGroups, billInfo, ruleInfo);
    }

    protected void setValue(BillInfo billInfo, List<RemarkFieldMetadata> cargoNameFieldMetadatas, List<RemarkFieldMetadata> itemSpecFieldMetadatas,
            int itemSpecNameLengthLimit, ItemGroup itemGroup, Integer cargoNameSeparatorType) {

        for (BillItem billItem : itemGroup.getBillItems()) {

            String joinName;
            if (cargoNameSeparatorType != null) {
                joinName = parseFieldValue(cargoNameFieldMetadatas, billInfo, billItem, cargoNameSeparatorType);
            } else {
                joinName = parseFieldValue(cargoNameFieldMetadatas, billInfo, billItem);
            }

            if (StringUtils.hasText(joinName)) {
                billItem.setJoinName(joinName);
            }

            String itemSpecName = parseFieldValue(itemSpecFieldMetadatas, billInfo, billItem);
            if (gtLimit(itemSpecName, itemSpecNameLengthLimit)) {
                itemSpecName = CommonTools.substring(itemSpecName, itemSpecNameLengthLimit, GBK);
            }
            if (StringUtils.hasText(itemSpecName)) {
                billItem.setItemSpec(itemSpecName);
            }
        }
    }

    /**
     * 万科商品名称定制版 根据配置取出字段值，若有则拼接符号
     * @param fieldMetaDatas
     * @param billInfo
     * @param billItem
     * @param cargoNameSeparatorType
     * @return
     */
    protected String parseFieldValue(List<RemarkFieldMetadata> fieldMetaDatas, BillInfo billInfo, BillItem billItem, Integer cargoNameSeparatorType) {

        if (CollectionUtils.isEmpty(fieldMetaDatas)) {
            return "";
        }

        /**
         * 1.取出所有值
         * 2.中间拼接符号
         */
        List<String> values = new ArrayList<>();
        for (int i = 0; i < fieldMetaDatas.size(); i++) {

            RemarkFieldMetadata fieldMetadata = fieldMetaDatas.get(i);
            String fieldName = fieldMetadata.getFieldName();
            Object obj = null;
            if (isBillInfoField(fieldMetadata)) {
                obj = billInfo;
            }
            if (isBillItemField(fieldMetadata)) {
                obj = billItem;
            }
            if (obj == null) {
                throw new SplitBizException("fieldGroupIndex value is illegal!");
            }
            String fieldValue = getCustomFieldValue(obj, fieldName);
            values.add(fieldValue);
        }

        /**
         * 拼接符号前判断这个值能否取到，若取不到则不拼接
         * 拼接符号条件：非第一个且当前值存在
         */
        String result = values.stream().filter(e -> !StringUtils.isEmpty(e)).collect(Collectors.joining(concatSeparator(cargoNameSeparatorType)));
        return result;
    }

    protected String parseFieldValue(List<RemarkFieldMetadata> cargoNameFieldMetadatas, BillInfo billInfo, BillItem billItem) {
        if (CollectionUtils.isEmpty(cargoNameFieldMetadatas)) {
            return "";
        }
        StringBuilder sb = new StringBuilder();
        for (RemarkFieldMetadata fieldMetadata : cargoNameFieldMetadatas) {
            String fieldName = fieldMetadata.getFieldName();
            Object obj = null;
            if (isBillInfoField(fieldMetadata)) {
                obj = billInfo;
            }
            if (isBillItemField(fieldMetadata)) {
                obj = billItem;
            }
            if (obj == null) {
                throw new SplitBizException("fieldGroupIndex value is illegal!");
            }
            Object fieldValue = null;
            try {
                fieldValue = FieldUtils.getFieldValue(obj, fieldName);
            } catch (ReflectionException ignore) {
                logger.warn(String.format("field = [%s] not found from obj = [%s]!", fieldName, obj.getClass()));
            }
            sb.append(Objects.toString(fieldValue, ""));
        }
        return sb.toString();
    }

    protected String getCustomFieldValue(Object object, String fieldName) {

        String text = "";
        if (object instanceof BillItem) {
            text = "明细信息";
        } else if (object instanceof BillInfo) {
            text = "主信息";
        }

        try {
            return Objects.toString(FieldUtils.getFieldValue(object, fieldName), "");
        } catch (ReflectionException e) {
            logger.warn("元数据获取{}字段值错误:{}", text, e.getMessage());
        }

        return "";
    }

    private CharSequence concatSeparator(Integer cargoNameSeparatorType) {

        CharSequence delimiter = "";
        if (cargoNameSeparatorType != null) {

            switch (cargoNameSeparatorType) {
                case 0:
                    delimiter = SPACE;
                    break;
                case 1:
                    delimiter = TAB;
                    break;
                case 2:
                    delimiter = COMMA;
                    break;
            }
        }

        return delimiter;
    }

    private boolean gtLimit(String str, int lengthLimit) {
        return str.getBytes(GBK).length > lengthLimit;
    }

    private boolean isBillItemField(RemarkFieldMetadata remarkFieldMetadata) {
        return remarkFieldMetadata.getFieldGroupIndex() == FieldOfObj.BILL_ITEM;
    }

    private boolean isBillInfoField(RemarkFieldMetadata remarkFieldMetadata) {
        return remarkFieldMetadata.getFieldGroupIndex() == FieldOfObj.BILL_INFO;
    }

    public void setCargoNameLengthLimit(int cargoNameLengthLimit) {
        this.cargoNameLengthLimit = cargoNameLengthLimit;
    }

    public void setItemSpecNameLengthLimit(int itemSpecNameLengthLimit) {
        this.itemSpecNameLengthLimit = itemSpecNameLengthLimit;
    }
}
