package com.xforceplus.phoenix.split.service.dataflow.impl;

import com.google.common.base.Stopwatch;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;
import com.google.common.collect.MultimapBuilder;
import com.xforceplus.phoenix.split.constant.TaxDeviceType;
import com.xforceplus.phoenix.split.domain.ItemGroup;
import com.xforceplus.phoenix.split.domain.PreInvoiceLimit;
import com.xforceplus.phoenix.split.domain.RuleInfo;
import com.xforceplus.phoenix.split.domain.SplitGroupLimit;
import com.xforceplus.phoenix.split.model.BillInfo;
import com.xforceplus.phoenix.split.model.BillItem;
import com.xforceplus.phoenix.split.service.SplitRuleUtil;
import com.xforceplus.phoenix.split.service.dataflow.DataProcessPlugin;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 立邦拆票合并发票实现
 */
@Service
public class MergeBySplitFieldPlugin implements DataProcessPlugin {

    private Logger logger = LoggerFactory.getLogger(getClass());

    @Override
    public List<ItemGroup> processData(List<ItemGroup> itemGroups, BillInfo billInfo, RuleInfo ruleInfo) {
        Stopwatch stopwatch = Stopwatch.createStarted();

        if (!ruleInfo.getSplitRule().isMergeBySplitFiled()) {
            return itemGroups;
        }

        List<ItemGroup> result = Lists.newLinkedList();
        Map<String, Multimap<String, ItemGroup>> temp = new HashMap<>();

        for (ItemGroup itemGroup : itemGroups) {
            String parentGroupFlag = itemGroup.getParentGroupFlag();
            String invoiceOfGroupKey = itemGroup.getInvoiceOfGroupKey();

            Multimap<String, ItemGroup> invoices = temp.get(parentGroupFlag);
            if (invoices == null) {
                invoices = MultimapBuilder.linkedHashKeys().linkedListValues().build();
                temp.put(parentGroupFlag, invoices);
            }
            invoices.put(invoiceOfGroupKey, itemGroup);

        }

        for (Map.Entry<String, Multimap<String, ItemGroup>> entry : temp.entrySet()) {
            Multimap<String, ItemGroup> invoices = entry.getValue();
            result.addAll(merge(invoices, ruleInfo));
        }

        logger.info("mergeBySplitFieldPlugin handle elapsed time = {} ms, itemGroup size:{}",
                stopwatch.elapsed().toMillis(), itemGroups.size());
        return result;
    }

    @Override
    public List<ItemGroup> processData(List<ItemGroup> itemGroups, BillInfo billInfo, RuleInfo ruleInfo, TaxDeviceType taxDeviceType) {
        return processData(itemGroups, billInfo, ruleInfo);
    }

    private List<ItemGroup> merge(Multimap<String, ItemGroup> invoices, RuleInfo ruleInfo) {

        List<ItemGroup> result = Lists.newLinkedList();

        List<ItemGroup> toBeProcessItemGroup = new ArrayList<>();
        for (String key : invoices.keySet()) {
            if (invoices.get(key).size() > 1) {
                result.addAll(invoices.get(key));
            } else {
                toBeProcessItemGroup.addAll(invoices.get(key));
            }
        }

        if (toBeProcessItemGroup.size() > 1) {
            // if (toBeProcessItemGroup.size() > 1 && toBeProcessItemGroup.size() <= 20) {
            toBeProcessItemGroup = mergerInvoice(toBeProcessItemGroup, ruleInfo);
        }

        result.addAll(toBeProcessItemGroup);
        return result;
    }

    private List<ItemGroup> mergerInvoice(List<ItemGroup> toBeProcessItemGroup, RuleInfo ruleInfo) {

        // SplitGroupLimit splitGroupLimit = SplitRuleUtil.createSplitGroupLimit(ruleInfo.getSplitRule(),
        //         toBeProcessItemGroup.get(0).getBillItems().get(0).getGoodsTaxNo(),
        //         toBeProcessItemGroup.get(0).getBillItems().get(0).getItemTypeCode());

        SplitGroupLimit splitGroupLimit = SplitRuleUtil.createSplitGroupLimit(
                ruleInfo.getSplitRule(),
                toBeProcessItemGroup.get(0).getBillItems().get(0).getItemTypeCode());

        List<List<BillItem>> invoices = greedyMergeInvoices(toBeProcessItemGroup, ruleInfo, splitGroupLimit);

        // MinInvoiceService minInvoiceService = new MinInvoiceService(splitGroupLimit);
        // List<List<BillItem>> invoices = minInvoiceService.minMergeInvoices(toBeProcessItemGroup);

        List<ItemGroup> result = Lists.newLinkedList();

        invoices.forEach(invoice -> {
            ItemGroup itemGroup = new ItemGroup();
            itemGroup.setBillItems(invoice);
            result.add(itemGroup);
        });
        return result;
    }

    /**
     * 使用贪心算法，拆分字段不能同时拆在两张票上
     * @param toBeProcessItemGroup
     * @param ruleInfo
     * @param splitGroupLimit
     * @return
     */
    private List<List<BillItem>> greedyMergeInvoices(List<ItemGroup> toBeProcessItemGroup, RuleInfo ruleInfo, SplitGroupLimit splitGroupLimit) {
        // 1. 对itemgroup倒排序
        // toBeProcessItemGroup = sortItemGroup(toBeProcessItemGroup, ruleInfo);

        List<List<BillItem>> invoices = new ArrayList<>();
        List<List<BillItem>> groupRemainInvoices = new ArrayList<>();

        for (ItemGroup itemGroup : toBeProcessItemGroup) {
            // List<BillItem> billItems = sortGroupBillItem(itemGroup.getBillItems(), ruleInfo);

            // 处理组内超限额情况
            List<BillItem> billItems = itemGroup.getBillItems();
            while (billItems.size() > 0) {
                List<BillItem> invoice = new ArrayList<>();
                PreInvoiceLimit limit = new PreInvoiceLimit(splitGroupLimit);
                for (int i = 0; i < billItems.size(); i++) {
                    if (limit.canAddItem(billItems.get(i))) {
                        invoice.add(billItems.get(i));
                        limit.addItem(billItems.get(i));
                    } else {
                        break;
                    }
                }

                billItems.removeAll(invoice);

                // group超限额
                if (billItems.size() != 0) {
                    invoices.add(invoice);
                } else {
                    groupRemainInvoices.add(invoice);
                }

            }

        }

        // 对每一组剩下的明细做汇总金额倒排序
        // groupRemainInvoices = sortGroupRemainInvoices(groupRemainInvoices, ruleInfo);

        while (groupRemainInvoices.size() > 0) {
            List<BillItem> invoice = new ArrayList<>();
            PreInvoiceLimit limit = new PreInvoiceLimit(splitGroupLimit);
            List<Integer> groupMergedIndexes = new ArrayList<>();

            for (int i = 0; i < groupRemainInvoices.size(); i++) {
                if (limit.canAddInvoice(groupRemainInvoices.get(i))) {
                    invoice.addAll(groupRemainInvoices.get(i));
                    limit.addInvoice(groupRemainInvoices.get(i));
                    groupMergedIndexes.add(i);
                } else {
                    break;
                }
            }

            invoices.add(invoice);

            int count = 0;
            for (Integer index : groupMergedIndexes) {
                groupRemainInvoices.remove(index - count);
                count++;
            }

        }

        return invoices;
    }

    /**
     * 对每组剩下的单独临时成票的倒排序
     */
    private List<List<BillItem>> sortGroupRemainInvoices(List<List<BillItem>> groupRemainInvoices, RuleInfo ruleInfo) {
        groupRemainInvoices = groupRemainInvoices.stream().sorted((o1, o2) -> {
            BigDecimal value1 = getGroupRemainBillItemSumAmount(o1, ruleInfo);
            BigDecimal value2 = getGroupRemainBillItemSumAmount(o2, ruleInfo);
            return value2.compareTo(value1);
        }).collect(Collectors.toList());

        return groupRemainInvoices;
    }

    /**
     * 对每组剩下的items求金额和
     * @param list
     * @param ruleInfo
     * @return
     */
    private BigDecimal getGroupRemainBillItemSumAmount(List<BillItem> list, RuleInfo ruleInfo) {
        if (ruleInfo.getSplitRule().isLimitIsAmountWithTax()) {
            return list.stream().map(BillItem::getAmountWithTax).reduce(BigDecimal.ZERO, BigDecimal::add);
        }
        return list.stream().map(BillItem::getAmountWithoutTax).reduce(BigDecimal.ZERO, BigDecimal::add);
    }

    /**
     * 对group 内的 billitem 排序
     * @param billItems
     * @return
     */
    private List<BillItem> sortGroupBillItem(List<BillItem> billItems, RuleInfo ruleInfo) {
        billItems = billItems.stream().sorted((o1, o2) -> {
            BigDecimal value1 = getBillItemAmount(o1, ruleInfo);
            BigDecimal value2 = getBillItemAmount(o1, ruleInfo);
            return value2.compareTo(value1);
        }).collect(Collectors.toList());

        return billItems;
    }

    /**
     * 获取billitem 金额, 需要区分含税不含税
     */
    private BigDecimal getBillItemAmount(BillItem item, RuleInfo ruleInfo) {
        if (ruleInfo.getSplitRule().isLimitIsAmountWithTax()) {
            return item.getAmountWithTax();
        }
        return item.getAmountWithoutTax();
    }

    /**
     * 对itemgroup倒排序, 需要区分含税不含税
     * @param itemGroups
     * @return
     */
    private List<ItemGroup> sortItemGroup(List<ItemGroup> itemGroups, RuleInfo ruleInfo) {
        itemGroups = itemGroups.stream().sorted((o1, o2) -> {
            BigDecimal value1 = getItemGroupBillItemSumAmount(o1, ruleInfo);
            BigDecimal value2 = getItemGroupBillItemSumAmount(o2, ruleInfo);
            return value2.compareTo(value1);
        }).collect(Collectors.toList());

        return itemGroups;
    }

    /**
     * 求每个itemgroup下billitems的不含税金额总和
     * @param itemGroup
     * @return
     */
    private BigDecimal getItemGroupBillItemSumAmount(ItemGroup itemGroup, RuleInfo ruleInfo) {
        if (ruleInfo.getSplitRule().isLimitIsAmountWithTax()) {
            return itemGroup.getBillItems().stream().map(BillItem::getAmountWithTax).reduce(BigDecimal.ZERO, BigDecimal::add);
        }
        return itemGroup.getBillItems().stream().map(BillItem::getAmountWithoutTax).reduce(BigDecimal.ZERO, BigDecimal::add);
    }
}
