package com.xforceplus.phoenix.split.service.dataflow.impl;

import com.xforceplus.phoenix.split.constant.InvoiceItemOrder;
import com.xforceplus.phoenix.split.constant.TaxDeviceType;
import com.xforceplus.phoenix.split.domain.ItemAmountInfo;
import com.xforceplus.phoenix.split.domain.ItemGroup;
import com.xforceplus.phoenix.split.domain.RuleInfo;
import com.xforceplus.phoenix.split.model.BillInfo;
import com.xforceplus.phoenix.split.model.BillItem;
import com.xforceplus.phoenix.split.model.SplitRule;
import com.xforceplus.phoenix.split.service.SplitBillItemAmountService;
import com.xforceplus.phoenix.split.service.SplitRuleUtil;
import com.xforceplus.phoenix.split.util.ThreadLocalFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.LinkedList;
import java.util.List;

import static com.xforceplus.phoenix.split.constant.RemarkConstant.HBBM_SET;

/**
 * 最少张数拆分
 * 每张票限额都拆满，比如限额100，有明细，101，102，则拆成100，1，99，3
 * <p>
 * 只支持按照不含税金额拆分
 */
@Service
public class MinSplitBillItemPlugin extends SplitBillItemPlugin {

    @Autowired
    @Qualifier("minInvoiceSplitItemServiceImpl")
    private SplitBillItemAmountService splitBillItemAmountService;

    /**
     * 发票上最小余额，余额小于等于0.99则另起发票
     */
    private static final BigDecimal MIN_BALANCE_LIMIT = new BigDecimal("0.99");

    /**
     * 1.按顺序处理明细
     * 2.当前明细是否需要拆分有一个记录当前票面剩余金额的变量X决定，当明细金额小于X时不需要拆分,大于X时拆分
     * 3.需拆分的明细先拆分一条X金额的明细，然后再判断是否大于限额，大于限额继续拆分直到不超限额
     * 4.处理完当前明细需要更新X值，当前明细不需要拆分则X等于X减明细金额，需要拆分X等于该条明细拆分出的最后一条明细金额减去限额的绝对值
     * @return
     */
    @Override
    public List<ItemGroup> processData(List<ItemGroup> itemGroups, BillInfo billInfo, RuleInfo ruleInfo, TaxDeviceType taxDeviceType) {

        final SplitRule rule = ruleInfo.getSplitRule();
        if (!InvoiceItemOrder.ITEM_NO_MINIMUM_INVOICES.value().equals(rule.getItemSort())) {
            return itemGroups;
        }

        itemGroups.forEach(itemGroup -> {

            List<BillItem> billItemList = new LinkedList<>();
            // String goodsTaxNo = itemGroup.getBillItems().get(0).getGoodsTaxNo();
            String itemTypeCode = itemGroup.getBillItems().get(0).getItemTypeCode();

            /**
             * balanceInvoiceAmount 表示一张发票上的剩余金额
             */
            BigDecimal invoiceRestLimitAmount = resetInvoiceLimitAmount(rule);
            int lineLimitNum = resetLineLimitNum(rule, itemTypeCode);

            for (BillItem billItem : itemGroup.getBillItems()) {

                checkPriceAndQuantity(billItem);

                mergeDiscountWithoutTax(billItem);

                /**
                 * 如果当前发票明细数量超限， 需要重置发票剩余金额和明细数， 另起一张发票
                 */
                if (exceedLineLimit(billItem, lineLimitNum)) {
                    lineLimitNum = resetLineLimitNum(rule, itemTypeCode);
                    invoiceRestLimitAmount = resetInvoiceLimitAmount(rule);
                }

                if (exceedInvoiceAmountLimit(billItem, invoiceRestLimitAmount)) {

                    List<BillItem> billItems = splitCurrentItem(billItem, rule, invoiceRestLimitAmount);
                    /**
                     * 拆分后只有最后一条明细不超限额，可以与其他明细一起成票，则发票剩余金额为限额减去这条明细的不含税金额
                     * 不用获取最小金额的，拆分时已经用剩余金额算好顺序
                     */
                    BillItem lastSplitItem = billItems.get(billItems.size() - 1);
                    // BillItem lastSplitItem = BillItemUtils.findMinAmountBillItem(billItems);

                    invoiceRestLimitAmount = rule.getInvoiceLimit().subtract(getActualAmount(lastSplitItem));
                    lineLimitNum = lineLimitNum - needLineNum(billItem);
                    billItemList.addAll(billItems);

                } else {

                    /**
                     * 不需拆分的明细不含税金额小于发票剩余金额，可以放进同一张发票
                     * 发票剩余金额等于当前剩余金额减去当前明细不含税金额
                     */
                    invoiceRestLimitAmount = invoiceRestLimitAmount.subtract(getActualAmount(billItem));
                    lineLimitNum = lineLimitNum - needLineNum(billItem);
                    billItemList.add(billItem);
                }

                /**
                 * 如果剩余限额小于发票最小金额，则需要重置发票限额，相等于另起一张发票
                 */
                if (isInvoiceAmountSaturated(invoiceRestLimitAmount)) {
                    invoiceRestLimitAmount = resetInvoiceLimitAmount(rule);
                    lineLimitNum = resetLineLimitNum(rule, itemTypeCode);
                }

            }
            itemGroup.setBillItems(billItemList);
        });

        return itemGroups;
    }

    private void mergeDiscountWithoutTax(BillItem billItem) {
        billItem.setDiscountWithoutTax(billItem.getOutterDiscountWithoutTax().add(billItem.getOutterPrepayAmountWithoutTax())
                .add(billItem.getInnerDiscountWithoutTax().add(billItem.getInnerPrepayAmountWithoutTax())));
    }

    private List<BillItem> splitCurrentItem(BillItem billItem, SplitRule rule, BigDecimal balanceInvoiceAmount) {

        ThreadLocalFactory.initThreadLocal(billItem);

        ItemAmountInfo itemAmountInfo = copyBillItemAmountInfo(billItem, rule);
        itemAmountInfo.setBalanceInvoiceAmount(balanceInvoiceAmount);

        List<ItemAmountInfo> itemAmountInfoList = splitBillItemAmountService
                .splitAmount(itemAmountInfo, rule);

        ThreadLocalFactory.clearContext();

        logger.debug("split id = {} itemAmount, result = {}", billItem.getSalesbillItemId(), itemAmountInfoList);
        return reCreateBillItem(itemAmountInfoList, billItem);
    }

    private boolean isInvoiceAmountSaturated(BigDecimal invoiceRestAmount) {
        return MIN_BALANCE_LIMIT.compareTo(invoiceRestAmount) > 0;
    }

    private BigDecimal getActualAmount(BillItem billItem) {
        return billItem.getAmountWithoutTax().subtract(billItem.getDiscountWithoutTax());
    }

    /**
     * 判断明细是否需要拆分,当明细的不含税金额比发票剩余金额大时需要拆分
     * @param billItem
     * @param invoiceRestAmount
     * @return
     */
    private boolean exceedInvoiceAmountLimit(BillItem billItem, BigDecimal invoiceRestAmount) {

        BigDecimal amountWithoutTax = billItem.getAmountWithoutTax();
        BigDecimal discount = billItem.getInnerDiscountWithoutTax()
                .add(billItem.getInnerPrepayAmountWithoutTax())
                .add(billItem.getOutterDiscountWithoutTax())
                .add(billItem.getOutterPrepayAmountWithoutTax());

        return amountWithoutTax.subtract(discount).compareTo(invoiceRestAmount) > 0;
    }

    /**
     * 是否超出行数限制
     * @param billItem
     * @param lineNum
     * @return
     */
    private boolean exceedLineLimit(BillItem billItem, int lineNum) {
        int needLineNum = needLineNum(billItem);

        if (needLineNum > lineNum) {
            return true;
        }
        return false;
    }

    private int resetLineLimitNum(SplitRule rule, String goodsTaxNo, String itemTypeCode) {

        return SplitRuleUtil.getLimitLine(rule, !HBBM_SET.contains(goodsTaxNo), itemTypeCode);
    }

    private int resetLineLimitNum(SplitRule rule, String itemTypeCode) {

        return SplitRuleUtil.getLimitLine(rule, itemTypeCode);
    }

    private BigDecimal resetInvoiceLimitAmount(SplitRule rule) {
        return rule.getInvoiceLimit();
    }
}
